<?php
namespace App\Core\Utilities;

use App\Core\Conn;
use App\Core\Helpers\Helper;
use Exception;

trait Create
{
    /**
     * Trait responsável por dar um insert na base de dados
     * 
     * @param array $attributes - recebe o nome da coluna + valor
     * @param ?string $url - url para redirecionamento pós execução
     * @param ?string $message_get - devolve uma informação via parâmetro get
     * @param ?bool $get_id - se true, devolve o id do insert
     * 
     * @return mixed
     * 
     * @todo Implementar Try Catch
     */
    public function create(
        array $attributes, 
        ?string $url = null, 
        ?string $message_get = '',
        ?bool $get_id = false
    ): mixed {
        $columns         = implode(',', array_keys($attributes));
        $interrogations  = substr(str_replace('?', '?,', str_repeat('?', count( array_keys($attributes) ))), 0, -1);
        $bind_types      = implode('', array_map(function($v) {

            if (is_string($v) || is_null($v)) {
                return 's';
            } else if (is_int($v)) {
                return 'i';
            } else if (is_double($v)) {
                return 'd';
            }

            return false;
        }, array_values($attributes)));

        $query  = "INSERT INTO {$this->table} ({$columns}) VALUES ({$interrogations})";
        $create = $this->conn->prepare($query);
        $create->bind_param($bind_types, ...array_values($attributes));

        if ($create->execute()) {

            if ($url) {
                $url = explode('?', $url);
                header("Location: " . $url[0] . $message_get);
            }

            if ($get_id) {
                return (int)$this->conn->insert_id;
            }

            return true;
        }

        $this->conn->rollback();
        return false;
    }


    /**
     * Trait para criar uma tabela se não existir
     * 
     * @param string $tableName - nome da tabela
     * @param array $columns - colunas e tipagem
     * 
     * @return bool
     * 
     * @todo implementar proteção contra SQL Injection
     */
    public function createTable(string $tableName, array $columns): bool {
        // Define a consulta SQL
        $sql = "CREATE TABLE IF NOT EXISTS $tableName (";
        // Adiciona as colunas à consulta SQL
        foreach ($columns as $columnName => $columnDefinition) {
            $sql .= "`$columnName` $columnDefinition, ";
        }

        // Remove a última vírgula e fecha a consulta SQL
        $sql = rtrim($sql, ', ') . ")";

        // Debug Query
        // throw new Exception("SQL: ". $sql, 400);

        // Cria um prepared statement
        $stmt = $this->conn->prepare($sql);
        // Verifica se ocorreu algum erro na criação do prepared statement
        if (!$stmt) {
            throw new Exception("Erro ao preparar statement: " . $this->conn->error);
        }

        // Executa o prepared statement
        $result = $stmt->execute();

        // Verifica se houve erro na criação da tabela
        if (!$result) {
            throw new Exception("Erro ao criar tabela: " . $stmt->error);
        }

        return true;
    }
}
