<?php

namespace App\Core\PDFTemplate;

// Inclui o arquivo de inicialização do TCPDF
require_once __DIR__ . '/../../../bootstrap.php';

use App\Core\Helpers\Helper;

class Orcamento extends \TCPDF
{
    /**
     * Instância do TCPDF
     * 
     * @var \TCPDF
     */
    private \TCPDF $pdf;

    /**
     * Dados do orçamento no PDF
     * 
     * @var array
     */
    private array $dados = [];

    /**
     * Construtor da classe
     * 
     * @param array $dados Dados do orçamento
     */
    public function __construct(array $dados)
    {
        $this->pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        $this->dados = $dados;
        $this->setConfigurations();
    }

    /**
     * Define as configurações do documento PDF
     * 
     * @return void
     */
    private function setConfigurations(): void
    {
        // Define as informações do documento
        $this->pdf->SetCreator(PDF_CREATOR);
        $this->pdf->SetAuthor('Fattos');
        $this->pdf->SetTitle('TBT Brindes - Orçamento'); // .env
        $this->pdf->SetSubject('Novo orçamento');
        $this->pdf->SetKeywords('TCPDF, PDF, example, test, guide');

        // Define a fonte para o cabeçalho e rodapé
        $this->pdf->setHeaderFont(array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));

        // Define a fonte padrão monoespaçada
        $this->pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

        // Define as margens do documento
        $this->pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $this->pdf->SetHeaderMargin(0);
        $this->pdf->SetFooterMargin(0);

        // Remove o rodapé padrão
        $this->pdf->setPrintFooter(false);

        // Define a quebra automática de página
        $this->pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // Define o fator de escala da imagem
        $this->pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);

        // Registrar a fonte Metropolis
        $this->pdf->AddFont('metropolis', '', __DIR__ . '/Config/TCPDF/metropolis.php');
    }

    /**
     * Renderiza o PDF
     */
    public function renderPDF(): array
    {
        // Adiciona uma página
        $this->pdf->AddPage();

        // Obtém a margem atual de quebra de página
        $bMargin = $this->pdf->getBreakMargin();

        // Obtém o modo atual de quebra automática de página
        $auto_page_break = $this->pdf->getAutoPageBreak();

        // Desativa a quebra automática de página
        $this->pdf->SetAutoPageBreak(false, 0);

        // Define a imagem de fundo
        $img_file = __DIR__ . '/../../view/assets/file/system/bgtbt.jpg';
        $this->pdf->Image($img_file, 0, 0, 210, 297, '', '', '', false, 300, '', false, false, 0);

        // Restaura o status de quebra automática de página
        $this->pdf->SetAutoPageBreak($auto_page_break, $bMargin);

        // Define o ponto inicial para o conteúdo da página
        $this->pdf->setPageMark();

        $this->pdf->SetFillColor(255, 255, 255);

        # Minha Empresa Session
        $this->pdf->writeHTMLCell(0, '', 15, 50, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">TBT Brindes e personalizados Ltda.</span></h1>'
            , 0, 0, 0, true, 'J', true
        );

        // ! .env
        $this->pdf->writeHTMLCell(0, '', 15, 55, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">CNPJ: 36.141.688/0001-77</span></h1>'
            , 0, 0, 0, true, 'J', true
        );

        // ! .env
        $this->pdf->writeHTMLCell(0, '', 15, 60, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">Cidade: Joinville - SC</span></h1>'
            , 0, 0, 0, true, 'J', true
        );

        # Cliente Session
        $nome_cliente = $this->dados['cliente']['pessoa_juridica'] == 1 ? 'Razão social: ' . $this->dados['cliente']['nome'] : 'Nome: ' . $this->dados['cliente']['nome'];

        //Razão social ou nome
        $this->pdf->writeHTMLCell(0, '', 15, 75, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">'. $nome_cliente .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Dados já formatados
        $documento_cliente = $this->dados['cliente']['pessoa_juridica'] == 1 ? 'CNPJ: ' . $this->dados['cliente']['cpf_cnpj'] : 'CPF: ' . $this->dados['cliente']['cpf_cnpj'];
        $documento_cliente = !empty($doc_client) ? $documento_cliente : 'CPF/CNPJ:';
        
        // CNPJ ou CPF
        $this->pdf->writeHTMLCell(0, '', 110, 75, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">'. $documento_cliente .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Endereço
        $this->pdf->writeHTMLCell(0, '', 15, 80, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Endereço: '. $this->dados['cliente']['endereco'] . ' - ' . $this->dados['cliente']['numero'] .' </span>'
            , 0, 0, 0, true, 'J', true
        );

        // Bairro
        $this->pdf->writeHTMLCell(0, '', 110, 80, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Bairro: '. $this->dados['cliente']['bairro'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // CEP
        $this->pdf->writeHTMLCell(0, '', 15, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">CEP: '. $this->dados['cliente']['cep'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Cidade
        $this->pdf->writeHTMLCell(0, '', 110, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Cidade: '. $this->dados['cliente']['cidade'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Estado
        $this->pdf->writeHTMLCell(0, '', 155, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Estado: '. $this->dados['cliente']['estado'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Contato
        $this->pdf->writeHTMLCell(0, '', 15, 90, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Contato: '. $this->dados['cliente']['pessoa'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // E-mail
        $this->pdf->writeHTMLCell(0, '', 110, 90, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">E-mail: '. $this->dados['cliente']['email'] .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // DIVISÓRIA
        $this->pdf->writeHTMLCell(0, '', 15, 100, 
            '<hr>'
            , 0, 0, 0, true, 'J', true
        );

        $product_html = '';
        $qtd_sum = 0;

        $issetPhoto = false;

        $valor_total = 0;
        foreach ($this->dados['produtos'] as $key => $value) {
            if ($value['foto'] != null) {
                $issetPhoto = true;
            }

            $nomeCortado = strlen($value['nome']) > 37 ? htmlspecialchars(substr($value['nome'], 0, 37)) . '...' : htmlspecialchars($value['nome']);

            $qtd_produto = (int) $value['qtd_produto'];
            $valor_total_orcado = (float) $value['valor_total_orcado'];
            $valor_unitario = $valor_total_orcado / $qtd_produto;
        
            // Arredondar o valor unitário para 2 casas decimais
            $valor_unitario_arredondado = round($valor_unitario, 2);
        
            // Recalcular o valor total para garantir consistência
            $valor_total_recalculado = $valor_unitario_arredondado * $qtd_produto;

            $valor_total += $valor_total_recalculado;
        
            $qtd_sum += $qtd_produto;
            $product_html .= '<tr>';
            $product_html .= '  <td style="font-size:10pt;">'. $qtd_produto .'</td>';
            $product_html .= '  <td style="font-size:10pt;">'. $nomeCortado . '</td>';
            $product_html .= '  <td style="font-size:10pt;">'. $value['ncm']. '</td>';
            $product_html .= '  <td style="font-size:10pt;" align="right"> R$ ' . number_format($valor_unitario_arredondado, 2, ',', '.') .' &nbsp;</td>';
            $product_html .= '  <td style="font-size:10pt; text-align:right;" align="right"> R$ ' . number_format($valor_total_recalculado, 2, ',', '.') . ' &nbsp;</td>';
            $product_html .= '</tr>';
        }

        if ($this->dados['juros'] > 0) {
            $valor_total += bcmul($valor_total, bcdiv($this->dados['juros'], 100, 4), 4);
        }

        if ($this->dados['desconto_valor'] > 0) {
            $valor_total -= $this->dados['desconto_valor'];
        }

        if ($this->dados['valor_frete'] > 0) {
            $valor_total += $this->dados['valor_frete'];
        }

        $n_parcela_info = '';
        if ($this->dados['n_parcela'] > 0) {
            $sum = bcdiv($valor_total, $this->dados['n_parcela'], 4);

            $com_juros = number_format($sum, 2, ',', '.');
            $n_parcela_info = $this->dados['n_parcela'] . 'x de R$' . $com_juros;
            $valor_total =  bcmul($sum, $this->dados['n_parcela'], 4);
        }

        $entrada_info = $this->dados['entrada_pgto'] > 0 ? 'R$ ' . number_format($this->dados['entrada_pgto'], 2, ',', '.') : 'Sem entrada';

        if ($this->dados['entrada_pgto'] > 0) {
            // $valor_total += $this->dados['entrada_pgto'];
        }

        $valor_total_return = $valor_total;

        $frete_info   = $this->dados['tipo_frete']== 'FOB' ? 'Cliente retira' : 'R$ ' . number_format($this->dados['valor_frete'] , 2, ',', '.');

        $th_desconto = '';
        $td_desconto = '';
        $cols_condicoes = 3;

        if ($this->dados['desconto_valor'] > 0) {
            $cols_condicoes = 4;
            $th_desconto = '<th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">DESCONTO</span></th>';
            $td_desconto = '<td style="font-size:10pt;">R$ ' . number_format($this->dados['desconto_valor'], 2, ',', '.') . '</td>';
        }

        $html = '
            <table align="center" border="1">
                <tr>
                    <th width="60"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">QTD.</span></th>
                    <th width="290"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">DESCRIÇÃO</span></th>
                    <th width="90"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">NCM</span></th>
                    <th width="90"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR UN.</span></th>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR</span></th>
                </tr>
                '. $product_html .'
            </table>
            <br /><br /><br />

            <table align="center" border="1">
                <tr>
                    <td colspan="' . $cols_condicoes . '"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">CONDIÇÕES DE PAGAMENTO</span></td>
                </tr>
                <tr>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">ENTRADA</span></th>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">FRETE</span></th>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">PARCELAS</span></th>
                    ' . $th_desconto . '
                </tr>
                <tr>
                    <td style="font-size:10pt;">' . $entrada_info. '</td>
                    <td style="font-size:10pt;">' . $frete_info . '</td>
                    <td style="font-size:10pt;">' . $n_parcela_info .'</td>
                    '. $td_desconto .'
                </tr>
            </table>
            <br /><br /><br />
            <table align="center" border="1">
                <tr>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">QTD. TOTAL</span></th>
                    <th width="431" rowspan="2"></th>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR TOTAL</span></th>
                </tr>
                <tr>
                    <th width="100" style="font-size:10pt;">'. $qtd_sum .'</th>
                    <th width="100" style="font-size:10pt;">R$ '. number_format($valor_total, 2, ',', '.') .'</th>
                </tr>
            </table>
            <br/><br/>
            <span style="font-size:9pt;">Prazo de entrega: </span><span style="color:#234A2D;font-size:9pt;">'. $this->dados['prazo_entrega'] .'</span>
            <br/>
            <span style="font-size:9pt;">Observações: </span><span style="color:#234A2D;font-size:9pt;"> '. $this->dados['observacao'].'</span>
        ';
    
        $this->pdf->writeHTMLCell(0, '', 15, 107, 
            $html
            , 0, 0, 0, true, 'J', true
        );

        // Data do orçamento - texto
        $this->pdf->writeHTMLCell(0, '', 20, 253, 
            '<span style="color:#234A2D;font-size:9pt;"><u>DATA DO ORÇAMENTO</u></span>'
            , 0, 0, 0, true, 'J', true
        );

        // Data do orçamento - data
        $this->pdf->writeHTMLCell(0, '', 30, 258, 
            '<span style="color:#234A2D;font-size:9pt;">'. date("d/m/Y") .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Se não existir nenhuma foto, gera o PDF e salva
        if (!$issetPhoto) {
            $path_save = __DIR__ . '/../../view/assets/file/budgets/';
            $name_file = Helper::generaterRandomString() . date('dmy') . '.pdf';
    
            // Fecha e gera o documento PDF
            $this->pdf->Output($path_save . $name_file, 'F'); // save in folder
            // $pdf->Output('ae.pdf', 'I'); // preview in browser
    
            return [
                'file' => $name_file,
                'valor_total' => $valor_total_return
            ];
        }

        // ADICIONANDO NOVA PÁGINA MANUALMENTE
        $this->pdf->AddPage();

        // Obtém a margem atual de quebra de página
        $bMargin = $this->pdf->getBreakMargin();

        // Obtém o modo atual de quebra automática de página
        $auto_page_break = $this->pdf->getAutoPageBreak();

        // Desativa a quebra automática de página
        $this->pdf->SetAutoPageBreak(false, 0);

        $this->pdf->Image($img_file, 0, 0, 210, 297, '', '', '', false, 300, '', false, false, 0);

        // Restaura o status de quebra automática de página
        $this->pdf->SetAutoPageBreak($auto_page_break, $bMargin);

        // Define o ponto inicial para o conteúdo da página
        $this->pdf->setPageMark();

        $this->pdf->SetFillColor(255, 255, 255);

        // Inicia a posição y onde as imagens serão colocadas
        $y = 55;

        // Inicia a posição x onde as imagens serão colocadas
        $x = 15;

        // Contador para as imagens
        $contador = 0;

        // Itera sobre o array de imagens
        foreach ($this->dados['produtos'] as $key => $value) {

            // Se não houver uma imagem, pula para a próxima
            if ($value['foto'] == null) {
                continue;
            }

            $file_url = __DIR__ . '/../../view/assets/file/product/' . $value['foto'];


            $name_product = strlen($value['nome']) > 32 ? mb_substr($value['nome'], 0, 32, 'UTF-8') . '...' : $value['nome'];

            // Adiciona a imagem ao PDF
            $this->pdf->writeHTMLCell(0, '', $x, $y, 
                '<span style="color:#234A2D;font-weight:bold;font-size:9pt;">'. $name_product .'</span><br>' .
                '<img src="'. $file_url . '" width="180" height="180">'
                , 0, 0, 0, true, 'J', true
            );

            // Incrementa o contador
            $contador++;

            // Se o contador é divisível por 3, inicia uma nova linha
            if ($contador % 3 == 0) {
                $y += 60; // 60 para a altura da imagem e 10 para o espaço entre as linhas
                $x = 15; // Reinicia a posição x para a margem esquerda
            } else {
                $x += 60; // 60 para a largura da imagem e 10 para o espaço entre as imagens
            }

            // Se o contador é divisível por 9, adiciona uma nova página
            if ($contador % 9 == 0) {
                $this->pdf->AddPage();
                // Reinicia a posição y para a margem superior
                $y = 55;
            }
        }

        $path_save = __DIR__ . '/../../view/assets/file/budgets/';
        $name_file = Helper::generaterRandomString() . date('dmy') . '.pdf';

        // Fecha e gera o documento PDF
        $this->pdf->Output($path_save . $name_file, 'F'); // save in folder
        // $pdf->Output('ae.pdf', 'I'); // preview in browser

        return [
            'file' => $name_file,
            'valor_total' => $valor_total_return
        ];
    }
}