<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, Insert, PreparedQuery, Update};
use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\{Controller, RenderTable};
use App\Core\Helpers\Helper;
use App\Core\Exceptions\PermissionException;

/**
 * Em breve irei migrar os dados do Budgets.php para cá
 */
class Orcamentos extends Controller
{
    use Create, Insert, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {        
        parent::__construct();
        $this->table = 'orcamentos';
    }

    /**
     * Método responsável por obter os dados do orçamento
     * 
     * @param int $idOrcamento
     * @return array
     */
    private function getDadosOrcamento(int $idOrcamento): array
    {
        $tblOrcamentosHistorico = RenderTable::table('orcamentos_historico');

        $query = 
            "SELECT 
                oh.* 
            FROM $tblOrcamentosHistorico AS oh
            WHERE oh.id_orcamento_FK = ? 
            ORDER BY oh.id DESC";

        $dados = self::PreparedQuery($query, [$idOrcamento])[0];

        return $dados;
    }

    public function listaOrcamentosCalendar(?string $dataInicio = null, ?string $dataFinal = null)
    {
        if (!empty($dataInicio) && !empty($dataFinal)) {
            $dataInicio = date('Y-m-d', strtotime($dataInicio));
            $dataFinal = date('Y-m-d', strtotime($dataFinal));
        }

        if (empty($dataInicio) || empty($dataFinal)) {
            $dataInicio = date('Y-m-01'); // Primeiro dia do mês atual
            $dataFinal = date('Y-m-t'); // Último dia do mês atual
        }

        $whereClausule = $_SESSION['user']['nivel'] == 1 ? ' AND o.create_by = ' . $_SESSION['user']['id'] : '';

        $getDiasPerder = (new Config)->getUltimoDiaTempoPerda();
        $dataDinamica  = date('Y-m-d', strtotime("-$getDiasPerder days"));

        $SQL = 
            "SELECT
                o.id,
                o.data_retorno AS start,
                CONCAT('ID: ', o.id, ', ', c.nome) AS title,
                '#c3c3c3' AS borderColor, -- isso aqui é desnecessário, posso fazer isso no JS
                CASE
                    WHEN o.status = -1 THEN '#e04a04'
                    WHEN o.status_user = 21 THEN '#206e09'
                    WHEN o.data_retorno = CURDATE() OR o.data_retorno = DATE_ADD(CURDATE(), INTERVAL 1 DAY) THEN '#ffcc00'
                    WHEN o.data_retorno < CURDATE() AND o.data_retorno >= '{$dataDinamica}' THEN '#c4c4c4'
                    WHEN o.data_retorno < '{$dataDinamica}' THEN '#91030a'
                    WHEN o.data_retorno < DATE_SUB(CURDATE(), INTERVAL 2 DAY) THEN '#0000FF'
                    ELSE NULL
                END AS backgroundColor
            FROM orcamentos AS o
            LEFT JOIN clientes AS c ON c.id = o.id_cliente_FK
            WHERE 1 = 1 
            $whereClausule
            AND (o.data_retorno BETWEEN ? AND ?)
            GROUP BY o.id, o.data_retorno, c.nome, o.status
            ORDER BY o.data_retorno ASC
        ";

        return self::PreparedQuery($SQL, [$dataInicio, $dataFinal]);
    }

    /**
     * Método responsável por retornar os produtos de um orçamento
     * 
     * @param int $id_orcamento - chave primária do orçamento
     * 
     * @return array $produtos - produtos do orçamento
     * 
     * @todo deverá obter através pelo id do histórico
     */
    public function getProductsById(int $idOrcamento): array
    {
        try {
            $tblOrcamentosProdutos = RenderTable::table('orcamentos_produtos');

            $query = 
                "SELECT 
                    op.id_produto_FK,
                    op.qtd, 
                    op.qtd_mao_obra, 
                    op.precificacao_nome,
                    op.valor_manual_unitario,
                    op.id_personalizacao_FK
                FROM $tblOrcamentosProdutos AS op
                WHERE op.id_orcamento_historico_FK = ?
            ";
    
            return self::PreparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 500);
        }
    }

    /**
     * Método responsável por obter a quantidade de edições de um orçamento
     * 
     * @param int $idOrcamento - chave primária do orçamento
     * 
     * @return int $total - total de edições
     */
    public function countHistoryBudget(int $idOrcamento): int
    {
        try {
            $tblOrcamentosHistorico = RenderTable::table('orcamentos_historico');

            $query = 
                "SELECT 
                    COUNT(*) AS total
                FROM $tblOrcamentosHistorico AS oh
                WHERE oh.id_orcamento_FK = ?
            ";
    
            $total = (int) self::PreparedQuery($query, [$idOrcamento])[0]['total'];

            return $total > 0 ? $total - 1 : 0;

        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 500);
        }
    }

    /**
     * Método responsável por reprovar o desconto de um orçamento
     * 
     * @param int $idOrcamento - chave primária do orçamento
     * 
     * @return bool
     */
    public function reprovarDesconto(int $idOrcamento): bool
    {
        try {
            $tblOrcamentos = RenderTable::table('orcamentos');
    
            # Somente Super Administradores tem permissão para aprovar um desconto
            if ($_SESSION['user']['nivel'] !== 9) {
                throw new PermissionException('Você não tem permissão para reprovar descontos');
            }

            self::updateQuery($tblOrcamentos, ['desconto_status' => 0], ['id' => $idOrcamento]);

            // (new Notification)->enviaNotificacaoDescontoReprovado($idOrcamento);

        } catch (PermissionException $pe) {
            Helper::jsonResponse(['error' => $pe->getMessage()], 403);
        } catch (\Throwable $th) {
            Helper::jsonResponse([
                'error' => 'Não foi possível reprovar o desconto, recarregue a página e tente novamente.',
                'error_system' => $th->getMessage()
            ], 500);
        }

        return true;
    }

    /**
     * Método responsável por aprovar o desconto de um orçamento
     * 
     * @param int $idOrcamento - chave primária do orçamento
     * 
     * @return bool
     */
    public function aprovarDesconto(int $idOrcamento): bool
    {        
        try {
            $tblOrcamentos = RenderTable::table('orcamentos');
    
            # Somente Super Administradores tem permissão para aprovar um desconto
            if ($_SESSION['user']['nivel'] < 8) {
                throw new PermissionException('Você não tem permissão para aprovar descontos');
            }

            self::updateQuery($tblOrcamentos, ['desconto_status' => 1], ['id' => $idOrcamento]);

            (new Notification)->enviaNotificacaoDescontoAprovado($idOrcamento);

        } catch (PermissionException $pe) {
            Helper::jsonResponse(['error' => $pe->getMessage()], 403);
        } catch (\Throwable $th) {
            Helper::jsonResponse([
                'error' => 'Não foi possível aprovar o desconto, recarregue a página e tente novamente.',
                'error_system' => $th->getMessage()
            ], 500);
        }

        return true;
    }
    
        /**
     * Método responsável por atualizar o status do orçamento
     * 
     * @param int $id_orcamento
     * @param int $status
     */
    public function updateStatus(int $id_orcamento, int $status)
    {
        try {
            self::updateQuery(
                RenderTable::table('orcamentos'), 
                ['status_user' => $status], 
                ['id' => $id_orcamento]
            );

            return true;
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }
    }

    public function getValorTotalOrcamentos(array $ids): array
    {
        try {
            $tblOrcamentosProdutos = RenderTable::table('orcamentos_produtos');
            $tblOrcamentosHistorico = RenderTable::table('orcamentos_historico');

            $ids = implode(',', $ids); // Converte o array de IDs em uma string

            $query = 
                "SELECT 
                    op.id_orcamento_FK, 
                    MAX(op.id_orcamento_historico_FK) as ultimo_id_historico,
                    SUM(op.valor_total_produto) + oh.frete as valor_total
                FROM $tblOrcamentosProdutos AS op
                JOIN $tblOrcamentosHistorico AS oh ON op.id_orcamento_historico_FK = oh.id
                WHERE op.id_orcamento_FK IN ($ids)
                GROUP BY op.id_orcamento_FK, op.id_orcamento_historico_FK
                ";

            return self::PreparedQuery($query);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 500);
        }
    }

    private function getQtdOrcamentosClientes(array $ids): array
    {
        try {
            $tblOrcamentos = RenderTable::table('orcamentos');

            $ids = implode(',', $ids); // Converte o array de IDs em uma string

            $query = 
                "SELECT 
                    COUNT(*) as qtd, 
                    id_cliente_FK
                FROM $tblOrcamentos
                WHERE id_cliente_FK IN ($ids)
                GROUP BY id_cliente_FK
                ";

            return self::PreparedQuery($query);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 500);
        }
    }

    public function obtemVendasPorVendedorMesAno(string $ano_mes)
    {
        try {
            $query = 
                "SELECT 
                    u.id,
                    COUNT(*) as qtd, 
                    SUM(o.valor_total) as valor_total,
                    u.nome as vendedor
                FROM orcamentos AS o
                LEFT JOIN usuarios AS u ON u.id = o.create_by
                WHERE 1 = 1
                    AND DATE_FORMAT(o.data_retorno, '%Y-%m') = ?
                    AND o.status_user = 21
                GROUP BY o.create_by
                ORDER BY valor_total DESC
            ";

            return self::PreparedQuery($query, [$ano_mes]);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 500);
        }
    }
    
    /**
     * @todo deverá também apagar da pasta os arquivos que foram deletados
     */
    public function salvaAnexosOrcamento(array $anexos, int $idOrcamento)
    {
        try {
            $reorganizedFiles = [];
            $fileKeys = ['name', 'full_path', 'type', 'tmp_name', 'error', 'size'];
        
            // Reorganiza os arquivos para que cada item fique no seu array correspondente
            foreach ($fileKeys as $key) {
                foreach ($anexos[$key] as $index => $value) {
                    $reorganizedFiles[$index][$key] = $value;
                }
            }
    
            $sql  = "DELETE FROM orcamentos_anexos WHERE id_orcamento_FK = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->bind_param('i', $idOrcamento);
    
            if (!$stmt->execute()) {
                throw new \Exception("Erro na execução da consulta: " . $stmt->error);
            }

            foreach ($reorganizedFiles as $anexo) {
                $name = FileManager::uploadFile('orcamento-anexos/', $anexo);

                self::Insert([
                    'id_orcamento_FK' => $idOrcamento,
                    'nome' => $name,
                ], 'orcamentos_anexos');
            }

            return ['success' => 'Anexos salvos com sucesso'];
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 500);
        }
    }

    public function getAnexosOrcamento(int $idOrcamento)
    {
        try {
            $query = 
                "SELECT 
                    oa.id,
                    oa.nome
                FROM orcamentos_anexos AS oa
                WHERE oa.id_orcamento_FK = ?
            ";

            return self::PreparedQuery($query, [$idOrcamento]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 500);
        }
    }

    public function obtemVendasDetalhadasPorVendedor(string $ano_mes, int $vendedor_id)
    {
        try {
            $query = 
                "SELECT 
                    o.id,
                    o.data_retorno,
                    o.valor_total,
                    c.nome as cliente,
                    cat.nome as origem
                FROM orcamentos AS o
                LEFT JOIN clientes AS c ON c.id = o.id_cliente_FK
                LEFT JOIN categoria AS cat ON cat.id = c.origem
                WHERE o.create_by = ?
                    AND DATE_FORMAT(o.data_retorno, '%Y-%m') = ?
                    AND o.status_user = 21
                ORDER BY o.data_retorno ASC
            ";

            return self::PreparedQuery($query, [$vendedor_id, $ano_mes]);
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 500);
        }
    }

    private function updateDataRetorno(int $idOrcamento, string $dataRetorno)
    {
        try {
            self::updateQuery(
                'orcamentos', 
                ['data_retorno' => $dataRetorno], 
                ['id' => $idOrcamento]
            );

            return true;
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }
    }

    /**
     * Método responsável por obter a rota do ajax e retornar a funçao
     * 
     * @param string $route nome da rota
     */
    private function route(string $route): void
    {
        $auth_middleware = new AuthenticationMiddleware;
        
        match ($route) {
            'aprovarDesconto' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->aprovarDesconto($_POST['idOrcamento']))
            ),
            'getDadosOrcamento' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getDadosOrcamento($_REQUEST['idOrcamento'])),
            ),
            'getProductsById' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getProductsById($_GET['id']))
            ),
            'countHistoryBudget' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->countHistoryBudget($_GET['id']))
            ),
            'updateStatus' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->updateStatus($_POST['id'], $_POST['status']))
            ),
            'getValorTotalOrcamentos' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getValorTotalOrcamentos($_POST['ids']))
            ),
            'getQtdOrcamentosClientes' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getQtdOrcamentosClientes($_REQUEST['ids']))
            ),
            'listaOrcamentosCalendar' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->listaOrcamentosCalendar($_POST['dataInicio'], $_POST['dataFinal']))
            ),
            'obtemVendasPorVendedorMesAno' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->obtemVendasPorVendedorMesAno($_REQUEST['ano_mes']))
            ),
            'obtemVendasDetalhadasPorVendedor' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->obtemVendasDetalhadasPorVendedor($_REQUEST['ano_mes'], $_REQUEST['vendedor_id']))
            ),

            'saveAttachments' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->salvaAnexosOrcamento($_FILES['anexos'], $_POST['idOrcamento']))
            ),

            'getAttachments' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->getAnexosOrcamento($_POST['idOrcamento']))
            ),

            'updateDataRetorno' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->updateDataRetorno($_POST['idOrcamento'], $_POST['dataRetorno']))
            ),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Orcamentos') {
    $instance = new Orcamentos();
    $instance->setRoute($_REQUEST['action']);
}