<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, RenderTable};
use App\Core\Helpers\Helper;
use Exception;

class Fees
{
    use Create, PreparedQuery, Update;

    private $conn = null;
    private string $table = '';
    private ?int $id_usuario = null;

    public function __construct()
    {
        $this->conn = Conn::openConn();
        $this->table = RenderTable::table('config_juros');
        $this->id_usuario = $_SESSION['user']['id'];
    }

    public function __destruct()
    {
        $this->conn->close();
        $this->conn = null;
    }

    private function store(int $vezes, float $juros)
    {
        if ($this->getCountVezes($vezes) >= 1) {
            Helper::jsonResponse(['error' => 'A quantidade informada já está cadastrada.'], 400);
        }

        try {
            self::Create([
                'vezes' => $vezes,
                'juros' => $juros,
                'created_by' => $this->id_usuario,
            ]);
        } catch (\Throwable $e) {
            Helper::jsonResponse('Erro ao cadastrar: ' . $e->getMessage(), 400);
        }

        Helper::jsonResponse(1);
    }

    /**
     *  Método privado responsável por obter a contagem de registros na tabela do sistema com o número específico de parcelas.
     *
     * @param int $vezes O número de parcelas a ser considerado na contagem.
     * @return int A quantidade de registros na tabela com o número específico de parcelas.
     */
    private function getCountVezes(int $vezes)
    {
        $data = self::preparedQuery("SELECT COUNT(*) AS qtd FROM {$this->table} WHERE vezes = ?", [$vezes]);
        return !empty($data[0]['qtd']) ? (int)$data[0]['qtd'] : 0;
    }

    /**
     * Método responsável por obter as taxas de um determinado registro na tabela do sistema.
     * @param int $id O ID do registro para o qual as taxas devem ser obtidas.
     */
    private function getFees(int $id)
    {
        try {
            $data = self::PreparedQuery("SELECT id, vezes, juros FROM {$this->table} WHERE id = ?", [$id]);
            Helper::jsonResponse($data[0]);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por atualizar os dados de um registro na tabela do sistema.
     * @param array $data Um array contendo os dados a serem atualizados.
     */
    private function update(array $data)
    {
        try {
            $this->validateIfExistsVezesOnUpdate($data);

            $fees = [
                'vezes' => (int)$data['vezes'],
                'juros' => Helper::formatFloatToAbsValue($data['juros']),
                'updated_by' => $this->id_usuario,
            ];

            $rowsAffected = self::updateQuery($this->table, $fees, ['id' => (int)$data['id']]);

            if ($rowsAffected === false) {
                Helper::jsonResponse("Erro ao atualizar.", 400);
            }

            if ($rowsAffected === 0) {
                Helper::jsonResponse("Nenhum valor foi atualizado.", 200);
            }

            $feesHistory = new FeesHistory();
            if (!$feesHistory->saveOldInfo($data)) {
                throw new Exception("Erro ao salvar histórico", 400);
            }

            Helper::jsonResponse('Os dados foram atualizados com sucesso.');
        } catch (\Throwable $e) {
            if (!$rowsAffected) {
                Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
            }
        }
    }

    private function validateIfExistsVezesOnUpdate(array $data)
    {
        $sql_verify = " SELECT CASE WHEN vezes = ? THEN 'vezes' WHEN vezes <> ? THEN NULL END AS duplicado ";
        $sql_verify .= " FROM " . RenderTable::table($this->table) . " ";
        $sql_verify .= " WHERE vezes = ? AND id <> ?";

        $code_product = self::PreparedQuery($sql_verify, [$data['vezes'], $data['vezes'], $data['vezes'], $data['id']]);
        if (!empty($code_product[0]['duplicado'])) {
            Helper::jsonResponse(['error' => 'A quantidade informada já está cadastrada.'], 400);
        }
    }

    private function delete(int $id)
    {
        try {
            $sql = "DELETE FROM {$this->table} WHERE id = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->bind_param('i', $id);
            $stmt->execute();

            if (!$stmt->execute()) {
                throw new Exception("Erro na execução da consulta: " . $stmt->error);
            }

            $stmt->close();
            Helper::jsonResponse(['success']);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function show(): array
    {
        return self::PreparedQuery("SELECT * FROM " . RenderTable::table('config_juros') . " ORDER BY vezes ASC");
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => $this->store(
                $_POST['vezes'],
                $_POST['juros']
            ),
            'get' => Helper::jsonResponse($this->show()),
            'getFees' => $this->getFees($_POST['idFees']),
            'update' => $this->update($_POST),
            'delete' => $this->delete($_POST['idFees']),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

// testar futuramente end(get_declared_classes())

if (isset($_POST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Fees') {
    $instance = new Fees();
    $instance->setRoute($_REQUEST['action']);
}
