<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, FileManager, RenderTable, Paginate};
use App\Core\Helpers\Helper;
use Exception;

class Customer
{
    use Create, PreparedQuery, Update;

    private $conn = null;
    private string $table = '';
    private ?int $id_usuario = null;

    public function __construct()
    {
        $this->conn = Conn::openConn();
        $this->table = RenderTable::table('clientes');
        $this->id_usuario = $_SESSION['user']['id'];
    }

    public function __destruct()
    {
        $this->conn->close();
        $this->conn = null;
    }

    /**
     * Método responsável por criar um novo usuário
     * 
     * @param array $info (recebe uma request com os dados do formulário)
     * @param ?array $photo (recebe a imagem do usuário)
     *
     */
    public function store(array $info, null|array $photo)
    {
        try {

            if ($this->clienteExistente($info['telefone'] ?? '', $info['celular'] ?? '')) {
                Helper::jsonResponse("Erro: Cliente já cadastrado.", 400);
            }
            
            if (isset($photo)) {
                $name_file = FileManager::uploadFile('customer/', $photo);
            }

            return self::Create([
                'nome' => $info['nome'],
                'email' => $info['email'],
                'cpf_cnpj' => !empty($info['cpfCnpj']) ? $info['cpfCnpj'] : null,
                'cep' => !empty($info['cep']) ? $info['cep'] : null,
                'bairro' => !empty($info['bairro']) ? $info['bairro'] : null,
                'endereco' => !empty($info['endereco']) ? $info['endereco'] : null,
                'cidade' => !empty($info['cidade']) ? $info['cidade'] : null,
                'estado' => !empty($info['estado']) ? $info['estado'] : null,
                'numero' => !empty($info['numero']) ? $info['numero'] : null,
                'origem' => $info['origemCliente'] ?? null,
                'complemento' => !empty($info['complemento']) ? $info['complemento'] : null,
                'insc_estadual' => !empty($info['inscricaoEstadual']) ? $info['inscricaoEstadual'] : null,
                'pessoa_juridica' => $info['pessoaJuridica'],
                'regime' => !empty($info['regimeTributario']) ? $info['regimeTributario'] : null,
                'telefone' => !empty($info['telefone']) ? $info['telefone'] : null,
                'celular' => !empty($info['celular']) ? $info['celular'] : null,
                'link_rede_social' => !empty($info['linkRedeSocial']) ? $info['linkRedeSocial'] : null,
                'rede_social' => !empty($info['redeSocial']) ? $info['redeSocial'] : null,
                'created_by' => $this->id_usuario,
                'logo' => $name_file ?? null
            ], null, '', true);

        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    private function clienteExistente(null|string $telefone = '', null|string $celular = ''): bool
    {
        $sql = "SELECT COUNT(*) AS qtd FROM " . RenderTable::table('clientes');
        $params = [];
    
        if (!empty($telefone)) {
            $sql .= " WHERE telefone = ?";
            $params[] = $telefone;
        }
    
        if (!empty($celular)) {
            if (!empty($params)) {
                $sql .= " OR celular = ?";
            } else {
                $sql .= " WHERE celular = ?";
            }
            $params[] = $celular;
        }
    
        if (empty($params)) {
            return false;
        }
    
        $result = self::PreparedQuery($sql, $params);
    
        if ($result[0]['qtd'] > 0) {
            return true;
        }
    
        return false;
    }

    private function filterCustomerLike(null|string $filter = '') {

        $sql_like = '';

        if ($filter) {
            $sql_like = 
                " AND (
                    c.nome LIKE '%$filter%' OR 
                    c.email LIKE '%$filter%' OR 
                    c.cpf_cnpj LIKE '%$filter%' OR 
                    c.telefone LIKE '%$filter%' OR 
                    c.celular LIKE '%$filter%' OR
                    u.nome LIKE '%$filter%' 
                )";
        }

        if (isset($_GET['vendedor']) && !empty($_GET['vendedor'])) {
            $vendedor = (int)$_GET['vendedor'];
            $sql_like .= " AND ( c.created_by = {$vendedor} )";
        }

        if (isset($_GET['origem']) && !empty($_GET['origem'])) {
            $origem = (int)$_GET['origem'];
            $sql_like .= " AND ( c.origem = {$origem} )";
        }

        if (isset($_GET['startDate']) && !empty($_GET['startDate'])) {
            $startDate = $_GET['startDate'];
            $sql_like .= " AND ( c.created_at >= '{$startDate} 00:00:00' )";
        }

        if (isset($_GET['endDate']) && !empty($_GET['endDate'])) {
            $endDate = $_GET['endDate'];
            $sql_like .= " AND ( c.created_at <= '{$endDate} 23:59:59' )";
        }

        return strtolower($sql_like);
    }

    /**
     * Método responsável por listar usuários do sistema
     *
     * @return array users
     */
    public function show(int $current_page, int $limit_per_page, null|string $filter = ''): array
    {
        $sql_like = $this->filterCustomerLike($filter ?? '');

        // $sql_is_vendedor = $_SESSION['user']['nivel'] == 1 ? " AND c.created_by = {$_SESSION['user']['id']}" : '';

        $paginate = new Paginate($this->getCountCustomer($sql_like), $current_page, $limit_per_page);

        $sql = 
            "SELECT 
                c.id, 
                c.nome, 
                c.cpf_cnpj, 
                c.pessoa_juridica, 
                LOWER(c.email) as email, 
                c.telefone, 
                c.celular,
                c.regime, 
                c.logo, 
                c.status, 
                c.created_by, 
                DATE_FORMAT(c.created_at, '%d/%m/%Y %H:%i') AS created_at, 
                CONCAT(
                    SUBSTRING_INDEX(u.nome, ' ', 1), 
                    ' ', 
                    SUBSTRING_INDEX(u.nome, ' ', -1)
                ) AS vendedor,
                cat.nome AS origem_name
            FROM clientes AS c
            LEFT JOIN usuarios AS u ON u.id = c.created_by
            LEFT JOIN categoria AS cat ON cat.id = c.origem 
            WHERE 1 = 1 $sql_like
            ORDER BY c.nome ASC
            LIMIT " . $paginate->getLimitOffset();
            
        return [
            'data' => self::PreparedQuery($sql),
            'pages' => $paginate->generatePaginationData($current_page)
        ];
    }

    /**
     * Método responsável por obter a contagem total de clientes cadastrados no sistema.
     * @return int A quantidade total de clientes cadastrados.
     */
    public function getCountCustomer(null|string $like = ''): int
    {
        # ! nada com nada
        $where = $like ? ' ' . $like : ' ' . $like;

        // $where .= $_SESSION['user']['nivel'] == 1 ? " AND c.created_by = {$_SESSION['user']['id']}" : '';
        
        $data = self::preparedQuery(
            "SELECT 
                COUNT(c.id) AS qtd 
            FROM {$this->table} AS c 
            LEFT JOIN usuarios AS u ON u.id = c.created_by
            WHERE 1 = 1 $where");
        return !empty($data[0]['qtd']) ? (int)$data[0]['qtd'] : 0;
    }

    /**
     * Método responsável por salvar alterações de determido cliente
     */
    public function update(array $data, null|array $photo)
    {
        try {
            $name_file = !empty($photo) ? FileManager::uploadFile('customer/', $photo) : (self::get($data['id'])['logo'] ?? null);

            // $this->validateIfExistsEmailOrCpfCpnjOnUpdate($data);

            $customer = [
                'nome' => $data['nome'],
                'email' => $data['email'],
                'cpf_cnpj' => !empty($data['cpfCnpj']) ? $data['cpfCnpj'] : null,
                'cep' => !empty($data['cep']) ? $data['cep'] : null,
                'bairro' => !empty($data['bairro']) ? $data['bairro'] : null,
                'endereco' => !empty($data['endereco']) ? $data['endereco'] : null,
                'cidade' => !empty($data['cidade']) ? $data['cidade'] : null,
                'estado' => !empty($data['estado']) ? $data['estado'] : null,
                'numero' => !empty($data['numero']) ? $data['numero'] : null,
                'complemento' => !empty($data['complemento']) ? $data['complemento'] : null,
                'insc_estadual' => !empty($data['inscricaoEstadual']) ? $data['inscricaoEstadual'] : null,
                'pessoa_juridica' => $data['pessoaJuridica'],
                'regime' => $data['regimeTributario'],
                'telefone' => !empty($data['telefone']) ? $data['telefone'] : null,
                'celular' => !empty($data['celular']) ? $data['celular'] : null,
                'link_rede_social' => !empty($data['linkRedeSocial']) ? $data['linkRedeSocial'] : null,
                'rede_social' => !empty($data['redeSocial']) ? $data['redeSocial'] : null,
                'logo' => $name_file ?? null,
                'origem' => $data['origemCliente'] ?? null,
                'created_by' => !empty($data['idVendedor']) ? (int)$data['idVendedor'] : $this->id_usuario, // 'created_by' => (int)$_SESSION['user']['id'], // 'created_by' => (int)$data['idVendedor'],
                'updated_by' => $this->id_usuario,
            ];

            self::updateQuery($this->table, $customer, ['id' => (int)$data['id']]);

            Helper::jsonResponse('Os dados foram atualizados com sucesso.');
        } catch (\Throwable $e) {
            // if (!$rowsAffected) {
                Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
            // }
        }
    }

    private function validateIfExistsEmailOrCpfCpnjOnUpdate(array $data)
    {
        $sql_verify_email = " SELECT CASE WHEN email = ? THEN 'email'";
        $sql_verify_email .= " WHEN EXISTS (SELECT 1 FROM clientes WHERE email = ?) THEN 'email'";
        $sql_verify_email .= " ELSE NULL END AS duplicado FROM " . RenderTable::table($this->table);
        $sql_verify_email .= " WHERE email = ? AND (id <> ?)";

        $sql_verify_email_result = self::PreparedQuery($sql_verify_email, [$data['email'], $data['email'], $data['email'], $data['id']]);

        if (!empty($sql_verify_email_result[0]['duplicado'])) {
            Helper::jsonResponse("Erro: E-mail já cadastrado.", 400);
        }

        $sql_verify_cpf_cnpj = " SELECT CASE WHEN cpf_cnpj = ? THEN 'cpfCnpj'";
        $sql_verify_cpf_cnpj .= " ELSE NULL END AS duplicado FROM " . RenderTable::table($this->table);
        $sql_verify_cpf_cnpj .= " WHERE cpf_cnpj = ? AND (cpf_cnpj IS NULL OR cpf_cnpj <> '') AND id <> ?";

        $sql_verify_cpf_cnpj_result = self::PreparedQuery($sql_verify_cpf_cnpj, [$data['cpfCnpj'], $data['cpfCnpj'], $data['id']]);

        if (!empty($sql_verify_cpf_cnpj_result[0]['duplicado'])) {
            Helper::jsonResponse("Erro: CPF/CNPJ já cadastrado.", 400);
        }
    }

    /**
     * Método responsável por alterar o status de um cliente no sistema.
     * @param int $id O ID do cliente a ser alterado.
     * @param int $status O novo status a ser atribuído ao cliente (0 para desativado, 1 para ativado).
     */
    public function changeCustomerStatus(int $id, int $status)
    {
        try {
            $status = ($status === 1) ? 0 : 1;

            $rowsAffected = self::updateQuery($this->table, ['status' => $status], ['id' => $id]);

            Helper::jsonResponse([
                'error' => 0,
                'status' => $status
            ]);

            if (!$rowsAffected) {
                throw new Exception("Erro ao alterar status." . 400);
            }
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function get(int $id)
    {
        $data = self::PreparedQuery("SELECT * FROM {$this->table} WHERE id = ?", [$id]);
        return $data[0];
    }

    public function getCustomer(int $id)
    {
        try {
            $data = self::PreparedQuery("SELECT * FROM clientes WHERE id = ?", [$id]);

            $data[0]['is_owner'] = 0;
            if ($data[0]['created_by'] == $_SESSION['user']['id'] || $_SESSION['user']['nivel'] > 7) {
                $data[0]['is_owner'] = 1;
            }

            Helper::jsonResponse($data[0]);
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    public function getNameAndId(): void
    {
        try {
            $sql_vendedor = $_SESSION['user']['nivel'] < 5 ? " AND created_by = {$_SESSION['user']['id']}" : '';
            Helper::jsonResponse(
                self::PreparedQuery("SELECT id, nome FROM {$this->table} WHERE 1 = 1 $sql_vendedor")
            );
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    private function getCountCustomersByMonth(): void
    {
        $sql  =  " SELECT MONTH(c.created_at) AS mes, COUNT(*) AS total_clientes, COUNT(c.convertido = 1 OR NULL) AS total_clientes_convertidos";
        $sql .=  " FROM " . RenderTable::table('clientes') . " AS c";
        $sql .=  " WHERE YEAR(c.created_at) = 2023 GROUP BY mes";
        $sql .=  " ORDER BY mes";

        $result = self::PreparedQuery($sql);

        $data = array();
        foreach($result as $row) {
            $data[] = array(
                "mes" => $row["mes"],
                "total_clientes" => $row["total_clientes"],
                "total_clientes_convertidos" => $row["total_clientes_convertidos"]
            );
        }

        Helper::jsonResponse($data);
    }

    private function getCountConvertedCustomerMonth(): void
    {
        $sql = " SELECT MONTH(created_at) as mes, COUNT(*) AS total_clientes";
        $sql .= " FROM " . RenderTable::table('clientes');
        $sql .= " WHERE YEAR(created_at) = 2023 and convertido = 1";
        $sql .= " GROUP BY mes ORDER BY mes";

        $result = self::PreparedQuery($sql);

        $data = array();
        foreach($result as $row) {
            $data[] = array(
                "mes" => $row["mes"],
                "total_clientes" => $row["total_clientes"]
            );
        }

        Helper::jsonResponse($data);
    }

    function countCustomersByUser(int $idUsuario): int
    {
        $sql = "SELECT COUNT(*) AS total_clientes FROM clientes WHERE created_by = ?";
        
        return self::PreparedQuery($sql, [$idUsuario])[0]['total_clientes'];
    }

    /**
     * Método responsável por transferir clientes e leads de um usuário para outro
     * 
     * @param int $idUsuarioDe ID do usuário que terá os clientes transferidos
     * @param int $idUsuarioPara ID do usuário que receberá os clientes
     * @param int $transferirLeads Se for 1, transferir leads
     * 
     * @return int Retorna a quantidade de clientes transferidos
     */
    function transferirClientes(int $idUsuarioDe, int $idUsuarioPara, int $transferirLeads = 0)
    {
        // Transferir leads
        if ($transferirLeads) {
            self::UpdateQuery('api_site', [
                'id_vendedor_FK' => $idUsuarioPara
            ], ['id_vendedor_FK' => $idUsuarioDe]);
        }

        // Transferir clientes
        return self::updateQuery('clientes', [
            'created_by' => $idUsuarioPara
        ], ['created_by' => $idUsuarioDe]);
    }

    public function getNameAndIdFiltered(string $search)
    {
        $where = $_SESSION['user']['nivel'] == 1 ? ' AND created_by = ' . $_SESSION['user']['id'] : '';

        $search = Helper::sanitizeStringParamSQL($search);
        
        try {
            return self::PreparedQuery(
                "SELECT id, nome, status, estado, telefone, celular 
                FROM {$this->table} 
                WHERE nome LIKE '%$search%' OR telefone LIKE '%$search%' OR celular LIKE '%$search%' 
                $where
                ORDER BY id DESC
            ");
            
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), $e->getCode() ?? 400);
        }
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     * @param string $route
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => Helper::jsonResponse($this->store(
                $_REQUEST,
                $_FILES['logo'] ?? null
            )),
            'getCustomer' => $this->getCustomer($_POST['idCustomer']),
            'update' => $this->update($_POST, $_FILES['logo'] ?? null),
            'changeCustomerStatus' => $this->changeCustomerStatus((int)$_POST['id'], (int)$_POST['status']),
            'getNameAndId' => $this->getNameAndId(),
            // 'getCountProspectCustomerMonth' => $this->getCountProspectCustomerMonth(),
            'getCountCustomersByMonth' => $this->getCountCustomersByMonth(),
            'countCustomersByUser' => Helper::jsonResponse($this->countCustomersByUser($_REQUEST['idUsuario'])),
            'transferirClientes' => Helper::jsonResponse($this->transferirClientes($_REQUEST['idUsuarioDe'], $_REQUEST['idUsuarioPara'])),
            'getNameAndIdFiltered' => Helper::jsonResponse($this->getNameAndIdFiltered($_POST['search'])),

        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Customer') {
    $instance = new Customer();
    $instance->setRoute($_REQUEST['action']);
}
