<?php

namespace App\Core;

require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Controller, Customer, RenderTable};
use App\Core\Helpers\Helper;

class Contact extends Controller
{
    use Create, PreparedQuery, Update;

    private string $table = '';

    public function __construct()
    {
        parent::__construct();
        $this->table = 'contatos';
    }

    public function store(array $data, array $files = [])
    {
        // Helper::jsonResponse([
        //     $files['anexo_item_0'],
        // ], 400);

        $this->conn->begin_transaction();


        try {

            $data = json_decode($data['params'], true);
            
            $name_files = [];

            foreach ($files as $key => $file) {
                if ($file == null) {
                    continue;
                }

                if (strpos($key, 'anexo_item_') !== false) {
                    continue;
                }

                $name_file = FileManager::uploadFile('contacts/', $file);
                array_push($name_files, $name_file);
            }        

            if (!is_numeric($data['cliente'])) {
                
                $info = [
                    'nome' => $data['cliente'],
                    'estado' => $data['estado'],
                    'cidade' => $data['cidade'],
                    'email' => '',
                    'origemCliente' => (int)$data['origem'],
                    'pessoaJuridica' => 0
                ];

                $telOrCel = strlen($data['tel']) < 15 ? 'telefone' : 'celular';

                $info[$telOrCel] = $data['tel'];

                $idCliente = (new Customer)->store($info, null);
            }

            $this->table = 'contatos';


            $pk = self::Create([
                'cliente' => is_numeric($data['cliente']) ? (int) $data['cliente'] : $idCliente,
                'contato' => $data['contato'],
                'tel' => $data['tel'],
                'origem' => (int)$data['origem'],
                'dialogo' => $data['dialogo'],
                'tipo_dialogo' => $data['tipoContatoRealizado'],
                'data_solicitada_entrega' => !empty($data['dataSolicitadaParaEntregar']) ? $data['dataSolicitadaParaEntregar'] : null,
                'destino_uso' => $data['destinoUso'] ?? '',
                'files' => $name_files ? json_encode($name_files) : null,
                'status_contato' => (int) $data['isSolicitacao'],
                // 'descricao_solicitacao' => $data['descricaoSolicitacao'] ?? null,
                'data_retorno' => !empty($data['dataRetorno']) ? $data['dataRetorno'] : null,
                'create_by' => $_SESSION['user']['id']
            ], null, '', true);

            $this->table = 'contatos_solicitacoes';
            foreach ($data['itensSolicitados'] as $item) {
                $anexoKey = 'anexo_item_' . $item['index'];
    
                self::Create([
                    'id_contato_FK' => $pk,
                    'qtd_solicitado' => $item['qtd'],
                    'descricao_solicitado' => $item['descricaoItem'],
                    'id_codigo_FK' => $item['tipoPersonalizacao'],
                    'valor_objetivo_solicitado' => !empty($item['valorObjetivo']) ? (float) $item['valorObjetivo'] : 0.00,
                    'anexo' => isset($files[$anexoKey]) ? FileManager::uploadFile('contacts/', $files[$anexoKey]) : null
                ]);
            }

            $this->conn->commit();
        } catch (\Throwable $e) {
            $this->conn->rollback();
            Helper::jsonResponse('Erro ao cadastrar: ' . $e->getMessage(), 400);
        }
    }

    public function getContatosParaRetornarCalendar()
    {
        $where_vendedor = $_SESSION['user']['nivel'] < '3' ? "AND ct.create_by = {$_SESSION['user']['id']}" : '';

        $query = 
            "SELECT 
                ct.id,
                ct.data_retorno AS start,
                cli.nome AS title,
                CASE 
                    WHEN ct.contato_retornado = 1 THEN '#206e09'
                    ELSE '#91030a'
                END AS backgroundColor
            FROM contatos AS ct
            LEFT JOIN clientes AS cli
                ON cli.id = ct.cliente
            WHERE ct.data_retorno IS NOT NULL
            $where_vendedor
        ";

        return self::PreparedQuery($query);
    }

    public function refOrcamento(int $id)
    {
        try {
            self::updateQuery('contatos', ['orcado' => 2], ['id' => $id]);
            
            return true;
        } catch (\Throwable $th) {
            throw new \Exception("Error Processing Request: " . $th->getMessage(), 400);
        }
    }

    public function getInfoById(int $id)
    {
        $query = 
            "SELECT 
                ct.*,
                c.nome AS nomeCliente
            FROM contatos AS ct
            LEFT JOIN clientes AS c
                ON c.id = ct.cliente
            WHERE ct.id = ?
        ";
        
        $data = self::PreparedQuery($query, [$id])[0];

        $data['fileZip'] = $data['files'] ? $this->compressZipFiles(json_decode($data['files'], true)) : '';

        // Decodifica o JSON da coluna 'produtos'
        $produtos = $data['produtos'] ? json_decode($data['produtos'], true) : [];
        
        if (!is_array($produtos)) {
            return $data;
        }

        // Cria um array somente com os IDs dos produtos
        $produtosIds = array_map(function($produto) {
            return intval($produto['produto']);
        }, $produtos);
        
        // Inicializa o array 'produtos'
        $data['produtos_list'] = [];

        // Se existem produtos, consulta a tabela 'produtos' para obter as informações dos produtos
        if ($produtosIds) {
    
            $produtosIds = implode(',', array_map('intval', $produtosIds));
            $queryProdutos = "SELECT id, nome FROM produtos WHERE id IN ($produtosIds)";
            $produtos = self::PreparedQuery($queryProdutos);
            $data['produtos_list'] = $produtos;
        }

        return $data;
    }


    public function getInfoByIdAndProducts(int $id)
    {
        $query = 
            "SELECT 
                ct.*,
                c.nome AS nomeCliente,
                CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS nomeVendedor
            FROM contatos AS ct
            LEFT JOIN clientes AS c
                ON c.id = ct.cliente
            LEFT JOIN usuarios AS u
                ON u.id = ct.create_by
            WHERE ct.id = ?
        ";
        
        $data = self::PreparedQuery($query, [$id])[0];

        $data['fileZip'] = $data['files'] ? $this->compressZipFiles(json_decode($data['files'], true)) : '';

        // Decodifica o JSON da coluna 'produtos'
        // $query 
        return [
            'data' =>$data,
            'produtos' => $this->getProdutosSolicitados($id)
        ];
    }

    public function getInfoByIdAndProductsRetorno(int $id)
    {
        $query = 
            "SELECT 
                ct.*,
                c.nome AS nomeCliente
            FROM contatos AS ct
            LEFT JOIN clientes AS c
                ON c.id = ct.cliente
            WHERE ct.id = ?
        ";
        
        $data = self::PreparedQuery($query, [$id])[0];

        $data['fileZip'] = $data['files'] ? $this->compressZipFiles(json_decode($data['files'], true)) : '';

        // Decodifica o JSON da coluna 'produtos'
        // $query 
        return [
            'data' =>$data,
            'produtos' => $this->getProdutosSolicitadosMergeRetornados($id)
        ];
    }

    public function getProdutosSolicitadosMergeRetornados(int $id)
    {
        $query = 
            "SELECT 
                cs.*,
                csr.id_produto_FK AS id_produto_retornado,
                csr.qtd_mao_de_obra AS qtdServicoRetornado,
                csr.id_personalizacao_FK AS idPersonalizacaoRetornado,
                p.nome AS nomeProdutoRetornado,
                pe.tipo AS nomePersonalizacao
            FROM contatos_solicitacoes_retornos AS csr
            LEFT JOIN contatos_solicitacoes AS cs
                ON cs.id = csr.id_contatos_solicitacoes_FK
            LEFT JOIN produtos AS p
                ON p.id = csr.id_produto_FK
            LEFT JOIN personalizacoes AS pe
                ON pe.id = cs.id_codigo_FK
            WHERE cs.id_contato_FK = ?
        ";

        return self::PreparedQuery($query, [$id]);
    }

    private function getProdutosSolicitados(int $id)
    {
        $query = 
            "SELECT 
                cs.*,
                p.tipo AS nomePersonalizacao
            FROM contatos_solicitacoes AS cs
            LEFT JOIN personalizacoes AS p
                ON p.id = cs.id_codigo_FK
            WHERE cs.id_contato_FK = ?
        ";

        return self::PreparedQuery($query, [$id]);
    }

    private function compressZipFiles(array $nameFiles): string
    {
        try {

            $zip = new \ZipArchive();
            $zipFileName = 'anexos.zip';
            $zipFilePath = __DIR__ . '/../../view/assets/file/contacts/' . $zipFileName;
        
            // Exclua o arquivo ZIP se ele existir
            if (file_exists($zipFilePath)) {
                unlink($zipFilePath);
            }
        
            if ($zip->open($zipFilePath, \ZipArchive::CREATE) === TRUE) {
                foreach ($nameFiles as $key => $fileName) {
                    $csvFilePath = __DIR__ . '/../../view/assets/file/contacts/' . $fileName;
                    
                    // Verifique se o arquivo existe antes de tentar adicioná-lo ao zip
                    if (file_exists($csvFilePath)) {
                        $zip->addFile($csvFilePath, $fileName);
                    }
                }
        
                $zip->close();
            }
        
            return $zipFileName;
        } catch (\Throwable $th) {
            // Você pode querer fazer algo com a exceção aqui, como registrar o erro
        }
    
        return '';
    }

    public function updateContactOrcamento(int $idContato)
    {
        try {
            self::updateQuery('contatos', ['orcado' => 1], ['id' => $idContato]);
            
            return true;
        } catch (\Throwable $th) {
            throw new \Exception("Error Processing Request: " . $th->getMessage(), 400);
        }
    }

    private function filterContactLike(
        null|string $filter = '', 
        $tipoFilter,
        $vendedorFilter,
        $startDate,
        $endDate,
        ?int $contatoId = null
    ) {

        $sql_like = $filter ? 
            " AND (cli.nome LIKE '%$filter%' OR ct.tel LIKE '%$filter%' OR ct.id LIKE '%$filter%')" : '';

        if ($tipoFilter == '1') {
            $sql_like .= " AND ct.latitude IS NOT NULL AND ct.longitude IS NOT NULL";
        }

        if ($tipoFilter == '2') {
            $sql_like .= " AND ct.latitude IS NULL AND ct.longitude IS NULL";
        }

        if ($vendedorFilter) {
            $sql_like .= " AND ct.create_by = $vendedorFilter";
        }

        if ($startDate) {
            $sql_like .= " AND DATE(ct.create_at) >= '$startDate'";
        }

        if ($endDate) {
            $sql_like .= " AND DATE(ct.create_at) <= '$endDate'";
        }

        if ($contatoId) {
            $sql_like .= " AND ct.id = $contatoId";
        }
       
        return $sql_like;
    }

    private function obtemOrderBy(?string $orderBy = '')
    {
        $dataAtual = date('Y-m-d');

        return match ($orderBy) {
            'id:desc' => ' ORDER BY ct.id DESC',
            'id:asc' => ' ORDER BY ct.id ASC',

            'dtRetorno:proximos' => " AND ct.data_retorno >= '{$dataAtual}' ORDER BY ct.data_retorno ASC ",

            default => ' ORDER BY ct.id DESC'
        };
    }


    /**
     * Método responsável por retornar informações do contato lançado
     * 
     * @return data
     * 
     * ! @todo remover asterístico do select 
     */
    public function show(int $current_page, int $limit_per_page = 10, null|string $filter = '')
    {
        $startDate = isset($_GET['startDate']) ? $_GET['startDate'] : '';
        $endDate = isset($_GET['endDate']) ? $_GET['endDate'] : '';
        $tipoFilter = isset($_GET['tipo']) ? $_GET['tipo'] : '';
        $vendedorFilter = isset($_GET['vendedor']) ? $_GET['vendedor'] : '';
        $contatoId = isset($_GET['contatoId']) ? (int) $_GET['contatoId'] : null;
        $orderBy = $this->obtemOrderBy($_GET['order'] ?? '');


        $sql_like = $this->filterContactLike($filter ?? '', $tipoFilter, $vendedorFilter, $startDate, $endDate, $contatoId);

        $paginate = new Paginate($this->getCountContacts($sql_like), $current_page, $limit_per_page);
        
        $query = 
            "SELECT 
                ct.*, -- // ! TEM QUE REMOVER ESSA PORCARIA DE ASTERÍSTICO
                DATE_FORMAT(ct.create_at, '%H:%i %d/%m/%Y') AS create_at,
                DATE_FORMAT(ct.data_retorno, '%d/%m/%Y') AS data_retorno,
                cli.nome AS nomeCliente,
                CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS nomeUsuario,
                u.img AS imgUsuario
            FROM 
                contatos AS ct
            LEFT JOIN clientes AS cli
                ON cli.id = ct.cliente
            LEFT JOIN usuarios AS u
                ON u.id = ct.create_by
            WHERE 1 = 1 $sql_like
            -- AND ct.id IN (
            --     SELECT MAX(id)
            --     FROM contatos
            --     GROUP BY cliente
            -- )
            {$orderBy}
            LIMIT {$paginate->getLimitOffset()}
        ";

        return [
            'data' => self::PreparedQuery($query),
            'pages' => $paginate->generatePaginationData($current_page, true)
        ]; 
    }

    public function getCountContacts(null|string $like = ''): int
    {
        $tblContatos = RenderTable::table('contatos');
        $tblClientes = RenderTable::table('clientes');

        $where = $like ? ' ' . $like : ' ' . $like;

        if (isset($_GET['order']) && $_GET['order'] == 'dtRetorno:proximos') {
            $dataAtual = date('Y-m-d');
            $where .= " AND ct.data_retorno >= '{$dataAtual}'";
        }


        $query = 
            "SELECT 
                COUNT(*) AS qtd
            FROM $tblContatos AS ct
            LEFT JOIN $tblClientes AS cli
                ON cli.id = ct.cliente
            WHERE 1 = 1 $where
        ";

        $data = self::preparedQuery($query);
        return !empty($data[0]['qtd']) ? (int)$data[0]['qtd'] : 0;
    }

    private function getDialogsByCustomer(int $idCustomer)
    {
        $tblContatos = RenderTable::table('contatos');

        $query = 
            "SELECT 
                ct.dialogo,
                DATE_FORMAT(ct.create_at, '%d/%m/%Y às %H:%i') AS create_at
            FROM $tblContatos AS ct
            WHERE ct.cliente = ?
            ORDER BY ct.id DESC
        ";

        return self::PreparedQuery($query, [$idCustomer]);
    }

    private function getClientNameContactId(int $idContact) 
    {
        $tblContatos = RenderTable::table('contatos');
        $tblClientes = RenderTable::table('clientes');

        $query = 
            "SELECT 
                c.nome AS nomeCliente
            FROM $tblContatos AS ct
            LEFT JOIN $tblClientes AS c
                ON c.id = ct.cliente
            WHERE ct.id = ?
        ";

        return self::PreparedQuery($query, [$idContact])[0];
    }

    public function showContatosCliente($idCliente)
    {
        $tblContatos = RenderTable::table('contatos');
        $tblClientes = RenderTable::table('clientes');
        $tblUsuarios = RenderTable::table('usuarios');

        $query = 
            "SELECT 
                ct.*,
                DATE_FORMAT(ct.create_at, '%H:%i %d/%m/%Y') AS create_at,
                cli.nome AS nomeCliente,
                u.nome AS nomeUsuario,
                u.img AS imgUsuario
            FROM 
                $tblContatos AS ct
            LEFT JOIN $tblClientes AS cli
                ON cli.id = ct.cliente
            LEFT JOIN $tblUsuarios AS u
                ON u.id = ct.create_by
            WHERE ct.cliente = ?
            ORDER BY ct.id DESC
        ";

        return self::PreparedQuery($query, [$idCliente]);
        
    }

    /**
     * Método responsável por retornar ações de contatos
     * 
     * @param array $ids
     * 
     * @return array
     * 
     * ! @TODO vulnerabilidade de SQL Injection para corrigir
     */
    private function getActions(array $ids = [])
    {
        $ids = implode(',', $ids);

        $query = "";

        if ($_SESSION['user']['nivel'] < 3) {
            $query = 
                "SELECT 
                    ct.id,
                    ct.status_contato
                FROM contatos AS ct
                WHERE 1 = 1 
                AND ct.id IN ($ids)
                AND (ct.status_contato = 1 OR ct.status_contato = 2)
                AND ct.orcado = 0
                AND ct.create_by = {$_SESSION['user']['id']}
            ";
        }

        if ($_SESSION['user']['nivel'] == 8 || $_SESSION['user']['nivel'] == 9) {
            $query = 
                "SELECT 
                    ct.id,
                    ct.status_contato
                FROM contatos AS ct
                WHERE 1 = 1 
                AND ct.id IN ($ids)
                AND (ct.status_contato = 1 OR ct.status_contato = 2)
                AND ct.orcado = 0
            ";
        }

        if (empty($query)) {
            return [];
        }

        return [
            'data' => self::PreparedQuery($query),
            'userLevel' => $_SESSION['user']['nivel']
        ];
    }

    private function updateContatoSolicitacaoOrcamento(
        int $idContato,
        null|array $produtos = NULL
    ) {

        $this->conn->begin_transaction();

        try {
            $this->table = 'contatos_solicitacoes_retornos';

            foreach ($produtos as $idSolicitacao => $info) {
                foreach ($info as $key => $item) {

                    if (!$item['produto']) {
                        continue;
                    }

                    $personalizacao = !empty($item['tipoPersonalizacao']) ? (int)$item['tipoPersonalizacao'] : null;
                    $qtdServicos = !empty($item['qtdServico']) ? (int)$item['qtdServico'] : 1;

                    self::Create([
                        'id_contato_FK' => $idContato,
                        'id_contatos_solicitacoes_FK' => $idSolicitacao,
                        'id_produto_FK' => (int) $item['produto'],
                        'id_personalizacao_FK' =>  $personalizacao,
                        'qtd_mao_de_obra' => $qtdServicos,
                        'created_by' => $_SESSION['user']['id'],
                    ]);
                }
            }

            self::updateQuery(
                'contatos', [
                    'atendido_em' => date('Y-m-d H:i:s'),
                    'status_contato' => 2
                ], ['id' => $idContato]
            );

            $this->conn->commit();
            return true;

        } catch (\Throwable $th) {
            $this->conn->rollback();
            throw new \Exception("Error Processing Request: " . $th->getMessage(), 400);
        }
    }

    public function saveContactRepresentante(array $data, array $files = [])
    {
        $name_files = [];

        foreach ($files as $key => $file) {
            if ($file == null) {
                continue;
            }

            $name_file = FileManager::uploadFile('contacts/', $file);
            array_push($name_files, $name_file);
        }        

        $info = [
            'nome' => $data['cliente'],
            'email' => '',
            'pessoaJuridica' => 0
        ];

        $telOrCel = strlen($data['tel']) < 15 ? 'telefone' : 'celular';

        $info[$telOrCel] = $data['tel'];

        $idCliente = (new Customer)->store($info, null);
        
        try {
            $this->table = RenderTable::table('contatos');

            self::Create([
                'cliente' => is_numeric($data['cliente']) ? (int) $data['cliente'] : $idCliente,
                'contato' => $data['contato'],
                'tel' => $data['tel'],
                'dialogo' => $data['dialogo'],
                'files' => $name_files ? json_encode($name_files) : null,
                // 'descricao_solicitacao' => $data['descricaoSolicitacao'] ?? null,
                'latitude' => $data['latitude'],
                'longitude' => $data['longitude'],
                'create_by' => $_SESSION['user']['id']
            ]);
        } catch (\Throwable $e) {
            Helper::jsonResponse('Erro ao cadastrar: ' . $e->getMessage(), 400);
        }
    }

    public function getGeolocalizacaoVisita(int $id)
    {
        $tblContatos = RenderTable::table('contatos');

        $query = 
            "SELECT 
                ct.latitude,
                ct.longitude
            FROM $tblContatos AS ct
            WHERE ct.id = ?
        ";

        return self::PreparedQuery($query, [$id])[0];
    }

    private function getCountContactsPorClientes(array $idsClientes)
    {
        $tblContatos = RenderTable::table('contatos');

        $idsClientes = implode(',', array_map('intval', $idsClientes));

        $query = 
            "SELECT 
                COUNT(*) AS qtd,
                cliente
            FROM $tblContatos
            WHERE cliente IN ($idsClientes)
            GROUP BY cliente
        ";

        return self::PreparedQuery($query);
    }

    private function atualizaStatusContatoRetorno(int $id, int $status)
    {
        try {
            self::updateQuery(RenderTable::table('contatos'), ['contato_retornado' => $status], ['id' => $id]);
            
            return true;
        } catch (\Throwable $th) {
            throw new \Exception("Error Processing Request: " . $th->getMessage(), 400);
        }
    }

    public function getUltimoContatoClientes(array $idsClientes)
    {
        $idsClientes = implode(',', array_map('intval', $idsClientes));

        $query = 
            "SELECT 
                ct.cliente AS idCliente,
                DATE_FORMAT(ct.create_at, '%H:%i %d/%m/%Y') AS create_at
            FROM contatos AS ct
            WHERE ct.cliente IN ($idsClientes)
            AND ct.id IN (
                SELECT MAX(id)
                FROM contatos
                WHERE cliente IN ($idsClientes)
                GROUP BY cliente
            )";

        return self::PreparedQuery($query);
    }

    /**
     * Obtem quantidade de contatos separados separado por mes filtrado pelo ano dos vendedores
     */
    public function obtemQtdContatosAnuais(int $ano)
    {
        try {
            $query = 
                "SELECT 
                    u.nome,
                    COUNT(*) AS qtd,
                    DATE_FORMAT(ct.create_at, '%m') AS mes
                FROM contatos AS ct
                LEFT JOIN usuarios AS u
                    ON u.id = ct.create_by
                WHERE 
                    DATE_FORMAT(ct.create_at, '%Y') = ?
                    AND u.tester = 0
                    AND u.nivel <> 9 AND u.nivel <> 8
                GROUP BY mes, u.id
            ";

            return self::preparedQuery($query, [$ano]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function obtemQtdOrigemContatosPorVendedorMesAno()
    {
        $yearMonth = $_REQUEST['ano_mes'] ?? date('Y-m');

        try {
            $query =
                "SELECT 
                    CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS nome,
                    COUNT(DISTINCT c.id) AS qtd, -- tem que ver esse distinct no futuro
                    IFNULL(cat.nome, 'Sem origem') AS origem,
                    DATE_FORMAT(c.created_at, '%m/%Y') AS mes_ano
                FROM contatos AS ct
                LEFT JOIN clientes AS c
                    ON c.id = ct.cliente
                LEFT JOIN usuarios AS u
                    ON u.id = ct.create_by
                LEFT JOIN categoria AS cat
                    ON cat.id = c.origem
                WHERE 1 = 1
                    AND u.tester = 0
                    AND u.nivel <> 9 AND u.nivel <> 8
                    AND DATE_FORMAT(ct.create_at, '%Y-%m') = ?
                GROUP BY u.id, c.origem
                ORDER BY qtd DESC
            ";

            return self::preparedQuery($query, [$yearMonth]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    public function getContatosPendentesSidebar()
    {
        if ($_SESSION['user']['nivel'] < 3) {

            $query = 
                "SELECT 
                    COUNT(*) AS qtd
                FROM contatos AS ct
                WHERE 1 = 1
                    AND (ct.status_contato = 2)
                    AND ct.orcado = 0
                    AND ct.create_by = {$_SESSION['user']['id']}
            ";

            return self::PreparedQuery($query)[0];
        }


        if ($_SESSION['user']['nivel'] == 8 || $_SESSION['user']['nivel'] == 9) {
            $query = 
                "SELECT 
                    COUNT(*) AS qtd
                FROM contatos AS ct
                WHERE 1 = 1
                    AND (ct.status_contato = 1)
                    AND ct.orcado = 0
            ";

            return self::PreparedQuery($query)[0];
        }
    }

    public function obtemQtdTipoContatoPorVendedorMesAno()
    {
        $yearMonth = $_REQUEST['ano_mes'] ?? date('Y-m');

        try {
            $query =
                "SELECT 
                    CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS nome,
                    COUNT(c.id) AS qtd, -- tem que ver esse distinct no futuro
                    ct.tipo_dialogo AS tipo,
                    DATE_FORMAT(c.created_at, '%m/%Y') AS mes_ano
                FROM contatos AS ct
                LEFT JOIN clientes AS c
                    ON c.id = ct.cliente
                LEFT JOIN usuarios AS u
                    ON u.id = ct.create_by
                WHERE 1 = 1
                    AND u.tester = 0
                    AND u.nivel <> 9 AND u.nivel <> 8
                    AND DATE_FORMAT(ct.create_at, '%Y-%m') = ?
                GROUP BY u.id, ct.tipo_dialogo
                ORDER BY qtd DESC
            ";

            return self::preparedQuery($query, [$yearMonth]);
        } catch (\Throwable $th) {
            Helper::jsonResponse(['error' => $th->getMessage()], 400);
        }
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     *
     * @param string $route
     *
     * @return void
     */
    private function route(string $route): void
    {
        match ($route) {
            'store' => $this->store($_POST, $_FILES),
            'saveContactRepresentante' => $this->saveContactRepresentante($_POST, $_FILES),
            'getInfoById' => Helper::jsonResponse($this->getInfoById($_GET['id'])),
            'getInfoByIdAndProducts' => Helper::jsonResponse($this->getInfoByIdAndProducts($_GET['id'])),
            'getInfoByIdAndProductsRetorno' => Helper::jsonResponse($this->getInfoByIdAndProductsRetorno($_GET['id'])),
            'getDialogsByCustomer' => Helper::jsonResponse($this->getDialogsByCustomer($_POST['idCustomer'])),
            'getClientNameContactId' => Helper::jsonResponse($this->getClientNameContactId($_GET['id'])),
            'getActions' => Helper::jsonResponse($this->getActions($_POST['ids'])),
            'atenderSolicitacaoOrcamentoContato' => Helper::jsonResponse($this->updateContatoSolicitacaoOrcamento($_POST['idContato'], $_POST['produtos'])),
            'getGeolocalizacao' => Helper::jsonResponse($this->getGeolocalizacaoVisita($_REQUEST['id'])),
            'getQtdContatosClientes' => Helper::jsonResponse($this->getCountContactsPorClientes($_REQUEST['ids'])),
            'getContatosParaRetornarCalendar' => Helper::jsonResponse($this->getContatosParaRetornarCalendar()),
            'atualizaStatusContatoRetorno' => Helper::jsonResponse($this->atualizaStatusContatoRetorno($_POST['idContato'], $_POST['status'])),
            'getUltimoContatoClientes' => Helper::jsonResponse($this->getUltimoContatoClientes($_REQUEST['ids'])),

            // Métricas vue 
            'obtemQtdContatosAnuais' => Helper::jsonResponse($this->obtemQtdContatosAnuais($_REQUEST['ano'])),
            'obtemQtdOrigemContatosPorVendedorMesAno' => Helper::jsonResponse($this->obtemQtdOrigemContatosPorVendedorMesAno()),
            'obtemQtdTipoContatoPorVendedorMesAno' => Helper::jsonResponse($this->obtemQtdTipoContatoPorVendedorMesAno()),
            // End Métricas

            // Sidebar
            'getContatosPendentesSidebar' => Helper::jsonResponse($this->getContatosPendentesSidebar()),
            // End Sidebar

            // Alpine Test (verificar depois)
            'show' => Helper::jsonResponse($this->show($_GET['page'], $_GET['limit'], $_GET['filter'] ?? '')),
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_REQUEST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Contact') {
    $instance = new Contact();
    $instance->setRoute($_REQUEST['action']);
}