<?php

namespace App\Core;

// Inclui o arquivo de inicialização do TCPDF
require_once __DIR__ . '/../../bootstrap.php';

use App\Core\Middlewares\AuthenticationMiddleware;
use App\Core\Utilities\{Create, PreparedQuery, Update};
use App\Core\{Conn, Config, Contact, Costs, Customer, RenderTable, Tax, Notification, Paginate, Product};
use App\Core\Helpers\Helper;
use App\Core\Interfaces\BudgetsInterface;
use App\Core\Precificacao\CalculoOrcamento;
use App\Core\Precificacao\Formulas\{
    CalculoDescontoOrcamento,
    CalculoFreteVendaOrcamento,
    CalculoJurosParcelamentoOrcamento,
    CalculoValorAdicionalOrcamento
};

use TCPDF;

class Budgets extends TCPDF implements BudgetsInterface
{
    use Create, PreparedQuery, Update;

    private $conn = null;

    private string $table = '';

    private bool $isUpdate = false;

    private string $dataDinamica = '';

    /**
     * Método responsável por fazer o calculo do novo valor final caso tenha adicionado um valor adicional
     * Será diluido o valor adicional nos produtos informados de forma proporcional ao valor final de cada produto já calculado
     * 
     * @param float $valorAdicional Valor informado para ser diluido entre os produtos
     * @param float $valorTotal Valor total de todos os produtos antes do adicional
     * @param array $produtos lista dos produtos
     */
    private function valorAdicional(float $valorAdicional, float $valorTotal, array $produtos)
    {
        foreach ($produtos as $key => $produto) {
            $soma = ((float) $produto['valor_total_orcado'] / $valorTotal) * $valorAdicional;
            $produtos[$key]['valor_total_orcado'] = (float) $produto['valor_total_orcado'] + $soma;
        }

        return $produtos;
    }

    /**
     * Método responsável por atualizar o valor total caso exista valor adicional informado pelo usuário
     * 
     * @param array $produtos
     * 
     * @return float valorTotal
     */
    private function calcTotalAdicional(array $produtos): float
    {
        $valorTotal = 0;
        foreach ($produtos as $produto) {
            $valorTotal += (float) $produto['valor_total_orcado']; 
        }

        return $valorTotal;
    }

    private static function mergeInfo(array $qtd_products_budgets) {
        $produtos = [];

        foreach ($qtd_products_budgets as $key => $produto) {
            $produtos[$key] = $produto;
            $produtos[$key]['valor_total_orcado'] = $produto['qtd_somado'];
            $produtos[$key]['qtd_produto'] = $produto['qtd'];
        }

        return $produtos;
    }

    private function calcTotal(
        array $produtos, 
        float $entrada = 0, 
        string $tipoEntrada, 
        float $frete = 0, 
        $valorAdicional = '',
        float $desconto = 0,
        string $tipoDesconto
    ) {
        $produto_total_somado = 0;
        $entrada_pgto = 0;
        $desconto_pgto = 0;
    
        foreach ($produtos as $key => $produto) {
            $produto_total_somado += (float)$produto['valor_total_orcado'];
        }
    
        $produto_total_somado = Helper::ensureEvenDecimal($produto_total_somado, function($number) {
            return Helper::ensureEvenDecimal($number);
        });

        // Helper::jsonResponse($produto_total_somado, 400);
    
        // Se existir valor adicional, efetuará o cálculo
        if (!empty($valorAdicional)) {
            $produtosMergeados = $this->valorAdicional((float)$valorAdicional, $produto_total_somado, $produtos);
            $produto_total_somado = $this->calcTotalAdicional($produtosMergeados);
        }
    
        if ($tipoDesconto == 'R$') {
            $desconto_pgto = $desconto;
            $produto_total_somado -= $desconto_pgto;
        } else if ($tipoDesconto == '%') {
            $desconto_pgto = Helper::truncateTwoDecimals(($desconto / 100) * $produto_total_somado);
            $produto_total_somado -= $desconto_pgto;
        }
        
        $produto_total_somado += Helper::ensureEvenDecimal($frete, function($number) {
            return Helper::ensureEvenDecimal($number);
        });

        if ($tipoEntrada == 'R$') {
            $entrada_pgto = $entrada;
        } else if ($tipoEntrada == '%') {
            $entrada = bcdiv($entrada, 100, 2);
            $entrada_pgto = Helper::truncateTwoDecimals(bcmul($entrada, $produto_total_somado, 3));
        }
        
        return [Helper::truncateTwoDecimals($produto_total_somado), Helper::truncateTwoDecimals($entrada_pgto), $produtosMergeados ?? $produtos, Helper::truncateTwoDecimals($desconto_pgto)];
    }
    
    /**
     * Método responsável por fazer um backup armazenando dados do orçamento vindo do cliente
     * Dados brutos, sem nenhum tipo de tratamento
     * 
     * É de extrema importância manter este método atualizado, pois será de importância para auditoria
     * 
     * @param array data - post data from client
     * 
     */
    private function storeBudgetBackup(int $id_orcamento, array $data)
    {
        $this->conn  = Conn::openConn();
        $this->table = 'orcamentos_historico';

        $n_parcelas  = null;
        $porcentagem = 0;

        # validar
        if (!empty($data['juros'])) {
            $n_parcelas  = (int) explode('-', $data['juros'])[0];
            $porcentagem = (float) explode('-', $data['juros'])[1];
        }

        try {
            $lasted_id =  self::Create([
                'id_orcamento_FK'   => $id_orcamento, // OBTER NO RETORNO DO SAVE
                'id_cliente_FK'     => $data['cliente'],
                'contato'           => $data['pessoa'],
                'data_retorno'      => $data['data_retorno'],
                'dialogo'           => $data['dialogo'],
                'n_parcelas'        => $n_parcelas,
                'porcentagem_juros' => $porcentagem,
                'entrada'           => (float) $data['entrada_pgto'],
                'entrada_tipo'      => $data['entrada_tipo'],
                'valor_adicional'   => (float) $data['valor_adicional'],
                'desconto'          => (float) $data['desconto_pgto'],
                'desconto_tipo'     => $data['desconto_tipo'],
                'prazo_entrega'     => $data['prazo'],
                'frete'             => (float) $data['valor_frete'],
                'frete_tipo'        => $data['tipo_frete'],
                'id_status_FK'      => $data['status_user'],
                'observacoes'       => $data['observacao'],
                'create_by'         => (int) $_SESSION['user']['id']
            ], null, '', true);

            return $lasted_id;

            return $lasted_id;
        } catch (\Throwable $th) {
            throw new \Exception("Erro ao cadastrar histórico do backup: " . $th->getMessage(), 400);
        }
    }

    private function update(
        int $id_orcamento,
        array $product, 
        int $id_cliente, 
        string $dialogo,
        string $data_retorno,
        float $valor_frete,
        string $tipo_frete,
        float $porcentagem_juros,
        int $parcelas,
        string $prazo_entrega = '',
        string $pessoa_contato = '',
        string $entrada_tipo,
        float $entrada_pgto = 0,
        string $observacao = '',
        float $valor_total,
        string $url_pdf,
        ?int $status_user = null,
        float $valor_adicional = 0,
        string $desconto_tipo,
        float $desconto_pgto = 0,
        float $desconto_valor = 0,
        int $idContact = 0,
        array $data
    ): void
    {
        $contact = null;

        # Se existir id de contato, significa que o contato está sendo transformado em orçamento, logo precisaremos atualizar o status do contato para orçado
        if ($idContact != 0) {
            $contact = $idContact;
            (new Contact)->updateContactOrcamento($idContact);
        }

        # envia notificação aos super administradores nv 9
        if ($desconto_pgto != 0) {
            # Categoria 3: Solicitação de desconto
            (new Notification)->sendNotificationSuperAdministradores(3);
        }

        $juros_calc = 0;
        if ($porcentagem_juros > 0) {
            $juros_calc = bcmul($valor_total, bcdiv($porcentagem_juros, 100, 4), 4);
            $valor_total += $juros_calc;
        }

        try {
            $costs = new Costs;
            $costs_values = $costs->get();
    
            $this->conn  = Conn::openConn();

            $this->table = RenderTable::table('orcamentos');

            $dataUpdate = [
                'id_cliente_FK'        => $id_cliente,
                'dialogo'              => $dialogo,
                'data_retorno'         => $data_retorno,
                'frete_valor'          => $valor_frete,
                'frete_tipo'           => $tipo_frete,
                'juros_valor'          => $juros_calc,
                'juros_porcentagem'    => $porcentagem_juros,
                'vezes_juros'          => $parcelas,
                'prazo_entrega'        => $prazo_entrega,
                'pessoa_contato'       => $pessoa_contato,
                'entrada_tipo'         => $entrada_tipo,
                'entrada_pgto'         => $entrada_pgto,
                'observacao_pedido'    => $observacao,
                'valor_total'          => $valor_total,
                'imposto'              => (float) $costs_values['imposto'],
                'custo_administrativo' => (float) $costs_values['custo_administrativo'],
                'comissao'             => (float) $costs_values['comissao'],
                'desconto_financeiro'  => (float) $costs_values['desconto_financeiro'],
                'status_user'          => $status_user, // Status cadastrado pelo usuário
                'file'                 => $url_pdf, // Caminho para o PDF
                'valor_adicional'      => $valor_adicional,
                'desconto'             => $desconto_pgto, // Aqui é o valor informado pelo usuário, ex: 10%
                'desconto_tipo'        => $desconto_tipo, // Aqui é o tipo de desconto % ou R$
                'desconto_status'      => $desconto_pgto != 0 ? -1 : NULL, // Se existir valor de desconto adiciona -1 senão adiciona null
                'desconto_valor'       => $desconto_valor, // Aqui já é o valor calculado, ex 10% de 200 = 20.00
                'contato_referencia'   => $contact // Se null, não é um orçamento vindo de um contato
            ];

            self::updateQuery('orcamentos', $dataUpdate, ['id' => $id_orcamento]);

            $idOrcamentoHistorico = $this->storeBudgetBackup((int)$id_orcamento, $data);

        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }



        try {
            $tax = new Tax;

            $this->table = 'orcamentos_produtos';

            foreach ($product as $key => $produto) {

                $valor_total_orcado = round($produto['valor_total_orcado'], 2);
                $valor_unitario = round($valor_total_orcado / $produto['qtd'], 4);         

                self::Create([
                    'id_orcamento_FK'     => (int) $id_orcamento,
                    'id_orcamento_historico_FK' => (int) $idOrcamentoHistorico,
                    'id_produto_FK'       => $produto['id'],
                    'id_personalizacao_FK' => !empty($produto['id_personalizacao']) ? (int) $produto['id_personalizacao'] : null,
                    'qtd_mao_obra'        => (int) $produto['qtd_mao_obra'],
                    'qtd'                 => (int) $produto['qtd_produto'],
                    'valor_total_produto' => $valor_total_orcado,
                    'precificacao_valor'  => $produto['precificacao_valor'],
                    'precificacao_nome'   => $produto['precificacao_nome'],
                    'porcentagem_comissao' => (float) $tax->getPorcentagemComissao($produto['precificacao_nome'])
                ]);
            }

        } catch (\Throwable $th) {
            // Se falhar, precisarei destruir o orcamento acima e o histórico do backup
            Helper::jsonResponse($th->getMessage() . 'ssss', 400);
        }

        $this->conn->close();
        $this->conn = null;
    }

    private function save(
        array $product, 
        int $id_cliente, 
        ?int $id_contato_FK, 
        string $dialogo,
        string $data_retorno,
        float $valor_frete,
        string $tipo_frete,
        float $porcentagem_juros,
        int $parcelas,
        string $prazo_entrega = '',
        string $pessoa_contato = '',
        string $entrada_tipo,
        float $entrada_pgto = 0,
        string $observacao = '',
        float $valor_total,
        string $url_pdf,
        ?int $status_user = null,
        float $valor_adicional = 0,
        string $desconto_tipo,
        float $desconto_pgto = 0,
        float $desconto_valor = 0,
        int $idContact = 0,
        array $data
    ): void
    {
        $contact = null;

        # Se existir id de contato, significa que o contato está sendo transformado em orçamento, logo precisaremos atualizar o status do contato para orçado
        if ($idContact != 0) {
            $contact = $idContact;
            (new Contact)->updateContactOrcamento($idContact);
        }

        # envia notificação aos super administradores nv 9
        if ($desconto_pgto != 0) {
            # Categoria 3: Solicitação de desconto
            (new Notification)->sendNotificationSuperAdministradores(3);
        }

        $juros_calc = 0;
        if ($porcentagem_juros > 0) {
            $juros_calc = bcmul($valor_total, bcdiv($porcentagem_juros, 100, 4), 4);
            // $valor_total += $juros_calc;
        }

        try {
            $costs = new Costs;
            $costs_values = $costs->get();
    
            $this->conn  = Conn::openConn();

            $this->table = RenderTable::table('orcamentos');

            $id_orcamento = self::Create([
                'id_cliente_FK'        => $id_cliente,
                'id_contato_FK'        => $id_contato_FK,
                'dialogo'              => $dialogo,
                'data_retorno'         => $data_retorno,
                'frete_valor'          => $valor_frete,
                'frete_tipo'           => $tipo_frete,
                'juros_valor'          => $juros_calc,
                'juros_porcentagem'    => $porcentagem_juros,
                'vezes_juros'          => $parcelas,
                'prazo_entrega'        => $prazo_entrega,
                'pessoa_contato'       => $pessoa_contato,
                'entrada_tipo'         => $entrada_tipo,
                'entrada_pgto'         => $entrada_pgto,
                'observacao_pedido'    => $observacao,
                'valor_total'          => $valor_total,
                'imposto'              => (float) $costs_values['imposto'],
                'custo_administrativo' => (float) $costs_values['custo_administrativo'],
                'comissao'             => (float) $costs_values['comissao'],
                'desconto_financeiro'  => (float) $costs_values['desconto_financeiro'],
                'status_user'          => $status_user, // Status cadastrado pelo usuário
                'file'                 => $url_pdf, // Caminho para o PDF
                'valor_adicional'      => $valor_adicional,
                'create_by'            => (int) $_SESSION['user']['id'],
                'desconto'             => $desconto_pgto, // Aqui é o valor informado pelo usuário, ex: 10%
                'desconto_tipo'        => $desconto_tipo, // Aqui é o tipo de desconto % ou R$
                'desconto_status'      => $desconto_pgto != 0 ? -1 : NULL, // Se existir valor de desconto adiciona -1 senão adiciona null
                'desconto_valor'       => $desconto_valor, // Aqui já é o valor calculado, ex 10% de 200 = 20.00
                'contato_referencia'   => $contact // Se null, não é um orçamento vindo de um contato
            ], null, '', true);

            $idOrcamentoHistorico = $this->storeBudgetBackup((int)$id_orcamento, $data);

        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }

        try {
            $tax = new Tax;

            $this->table = 'orcamentos_produtos';

            foreach ($product as $key => $produto) {

                $valor_total_orcado = round($produto['valor_total_orcado'], 2);
                $valor_unitario = round($valor_total_orcado / $produto['qtd'], 4);
            
                self::Create([
                    'id_orcamento_FK'     => (int) $id_orcamento,
                    'id_orcamento_historico_FK' => (int) $idOrcamentoHistorico,
                    'id_produto_FK'        => $produto['id'],
                    'id_personalizacao_FK' => !empty($produto['id_personalizacao']) ? (int) $produto['id_personalizacao'] : null,
                    'qtd_mao_obra'        => (int) $produto['qtd_mao_obra'],
                    'qtd'                 => (int) $produto['qtd_produto'],
                    'valor_total_produto' => $valor_total_orcado,
                    'precificacao_valor'  => $produto['precificacao_valor'],
                    'precificacao_nome'   => $produto['precificacao_nome'],
                    'valor_manual_unitario'  => isset($produto['valor_manual']) && !empty($produto['valor_manual']) ? (float) $produto['valor_manual'] : null,
                    'porcentagem_comissao' => (float) $tax->getPorcentagemComissao($produto['precificacao_nome'])
                ]);
            }

        } catch (\Throwable $th) {
            // Se falhar, precisarei destruir o orcamento acima e o histórico do backup
            Helper::jsonResponse($th->getMessage() . 'ssss', 400);
        }

        $this->conn->close();
        $this->conn = null;
    }

    /**
     * Mescla informações ao produto
     */
    private static function mergeProductAndTax(array $products, array $tax): array
    {
        $produtos = [];
        foreach ($products as $key => $produto) {
            $produtos[$key] = $produto;
            $produtos[$key]['precificacao_nome'] = $produto['precificacao'];
            $produtos[$key]['precificacao_valor'] = $tax[$key];
            $produtos[$key]['frete_compra'] = (float) $produto['frete_compra'];
            $produtos[$key]['frete_compra_regra'] = (float) $produto['frete_compra_regra'];
            $produtos[$key]['embalagem'] = (float) $produto['embalagem'];
            $produtos[$key]['embalagem_regra'] = (float) $produto['embalagem_regra'];
            $produtos[$key]['mao_obra'] = (float) $produto['mao_obra'];
            $produtos[$key]['mao_obra_regra'] = (float) $produto['mao_obra_regra'];
            $produtos[$key]['qtd'] = (int) $produto['qtd'];
            $produtos[$key]['qtd_mao_obra'] = (int) $produto['qtd_mao_obra'];
            $produtos[$key]['valor_compra'] = (float) $produto['valor_compra'];
            $produtos[$key]['valor_manual'] = isset($produto['valor_manual']) && !empty($produto['valor_manual']) ? (float) $produto['valor_manual'] : null;
            $produtos[$key]['extras'] =  (float) $produto['extras'];
        }

        return $produtos;
    }

    /**
     * Método responsável por mesclar o array de produtos a fim de incluir a quantidade (vem do front)
     * 
     * @param array $produtos produtos vindo da base
     * @param array $produtosFrond (dados do produtos vindo do front, incluindo a qtd)
     * 
     * @return array array mesclado
     */
    private function adicionaQtdProduto(array $produtos, array $produtosFront): array
    {
        $informacoesAssociativas = [];

        $count = 0;
        foreach ($produtosFront as $key => $front) {

            foreach ($produtos as $key => $produto) {
                if ($produto['id'] == $front['produto']) {
                    $informacoesAssociativas[$count. '-' . $front['produto']] = $produto;
                    $informacoesAssociativas[$count. '-' . $front['produto']]['qtd'] = $front['qtd'];
                    $informacoesAssociativas[$count. '-' . $front['produto']]['id_personalizacao'] = $front['id_personalizacao'];
                    $informacoesAssociativas[$count. '-' . $front['produto']]['qtd_mao_obra'] = $front['qtd_mao_obra'];
                    $informacoesAssociativas[$count. '-' . $front['produto']]['precificacao'] =  $front['precificacao'];
                    $informacoesAssociativas[$count. '-' . $front['produto']]['valor_manual'] = $front['valor_manual'];
                }
            }
            
            $count++;
        }

        return $informacoesAssociativas;
    }

    private function budgetPreview($produtos, $valorTotal, $valorEntrada, $informacoesCalculoDeJuros, $valorDescontado, $valorFrete)
    {
        Helper::jsonResponse([
            'preview' => 1,
            'valor_total' => $valorTotal,
            'produtos' => $produtos,
            'n_parcelas' => $informacoesCalculoDeJuros['NParcelas'],
            'valor_por_parcela' => $informacoesCalculoDeJuros['valorPorParcela'],
            'entrada' => $valorEntrada,
            'valor_frete' => $valorFrete,
            'valor_desconto' => $valorDescontado
        ]);
    }

    /**
     * Método responsável por crair um novo orçamento
     * 
     * @param array $data Informações do formulário cliente
     * @param bool $preview Parâmetro para pré-visualizar o orçamento (se true, não irá gravar na base, só irá retornar as informações para o cliente)  
     * 
     */
    public function store(array $data, bool $preview = false)
    {        
        $this->conn  = Conn::openConn();
        $this->table = 'orcamentos';

        $custos   = (new Costs)->calculate();
        $cliente  = (new Customer)->get($data['cliente']);
        $produtos = (new Product)->getDadosGeraisProdutos($data['produtos']);
        $produtos = $this->adicionaQtdProduto($produtos, $data['produtos']);
        $precificacao = (new Tax)->getTaxFromMultipleProducts($produtos);
        $produtos = $this->mergeProductAndTax($produtos, $precificacao);
        
        $valorFrete = !empty($data['valor_frete']) ? (float) $data['valor_frete'] : 0;
        $n_parcela   = (int) explode('-', $data['juros'])[0];
        $juros       = (float) explode('-', $data['juros'])[1];

        $calculoOrcamento = new CalculoOrcamento([
                'custos' => $custos,
                'juros' => $juros,
                'n_parcela' => $n_parcela,
                'valor_frete' => $valorFrete,
                'valor_adicional' => (float) $data['valor_adicional'],
                'desconto_pgto' => (float) $data['desconto_pgto'] ?? 0,
                'desconto_tipo' => $data['desconto_tipo'],
                'entrada_tipo' => $data['entrada_tipo'],
                'entrada_pgto' => (float) $data['entrada_pgto'],
            ],
            (new CalculoDescontoOrcamento()),
            (new CalculoFreteVendaOrcamento()), 
            (new CalculoJurosParcelamentoOrcamento()),
            (new CalculoValorAdicionalOrcamento()), 
        );

        $produtos = $calculoOrcamento->calculoDeHarvard($produtos); 
        list($produtos, $valorTotal, $valorEntrada, $informacoesCalculoDeJuros, $valorDescontado) = $calculoOrcamento->aplicarCalculosAdicionais($produtos, $data);

        // Retorna a pré-visualização do orçamento
        if ($preview) {
            $this->budgetPreview(
                $produtos, 
                $valorTotal, 
                $valorEntrada, 
                $informacoesCalculoDeJuros, 
                $valorDescontado,
                $valorFrete
            );
        }

        if (isset($data['ref']) && !empty($data['ref'])) {
            (new Contact)->refOrcamento($data['ref']);
        }

        try {
            
            $dataRender = $this->render(
                cliente:           $cliente['nome'],
                pessoa_juridica:   $cliente['pessoa_juridica'],
                doc_client:        $cliente['cpf_cnpj'],
                endereco_numero:   $cliente['endereco'] ?? ''. ', '. $cliente['numero'] ?? '',
                bairro:            $cliente['bairro'] ?? '',
                cep:               $cliente['cep'] ?? '',
                cidade:            $cliente['cidade'] ?? '', 
                estado:            $cliente['estado'] ?? '',
                pessoa_contato:    $data['pessoa'],
                email:             $cliente['email'],
                prazo_entrega:     $data['prazo'],
                valor_frete:       $data['valor_frete'],
                tipo_frete:        $data['tipo_frete'],
                produtos:          $produtosMergeadosComValorAdicionado ?? $produtosMergeados,
                valor_total:       $valor_total,
                entrada_pgto:      $entrada_pgto,
                n_parcela:         $n_parcela,
                juros:             $juros,
                observacao:        $data['observacao'],
                desconto_valor:    $desconto_pgto ?? 0
            );

            $file_name = $dataRender['file'];
            $valor_total = $dataRender['valor_total'];

        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), 400);
        }

        if ($this->isUpdate) {

            $this->update(
                $data['id_orcamento'],
                $produtosMergeadosComValorAdicionado ?? $produtosMergeados,
                $data['cliente'],
                $data['dialogo'],
                $data['data_retorno'],
                $valor_frete,
                $data['tipo_frete'],
                $juros,
                $n_parcela,
                $data['prazo'],
                $data['pessoa'],
                $data['entrada_tipo'],
                (float) $data['entrada_pgto'],
                $data['observacao'],
                $valor_total,
                $file_name,
                $data['status_user'],
                $data['valor_adicional'] ?? 0,
                $data['desconto_tipo'],
                (float)$data['desconto_pgto'] ?? 0,
                $desconto_pgto ?? 0,
                (int)$data['contactToBudget'],
                $data
            );

            return [
                'desconto_pgto' => (float) $data['desconto_pgto'],
                'pdf' => $file_name
            ];
        }

        try {

            $this->save(
                $produtosMergeadosComValorAdicionado ?? $produtosMergeados,
                $data['cliente'],
                $data['ref'] ?? null,
                $data['dialogo'],
                $data['data_retorno'],
                $valor_frete,
                $data['tipo_frete'],
                $juros,
                $n_parcela,
                $data['prazo'],
                $data['pessoa'],
                $data['entrada_tipo'],
                (float) $data['entrada_pgto'],
                $data['observacao'],
                $valor_total,
                $file_name,
                $data['status_user'],
                $data['valor_adicional'] ?? 0,
                $data['desconto_tipo'],
                (float)$data['desconto_pgto'] ?? 0,
                $desconto_pgto ?? 0,
                (int)$data['contactToBudget'],
                $data
            );
    
        } catch (\Throwable $e) {
            Helper::jsonResponse($e->getMessage(), 400);
        }

        Helper::jsonResponse([
            'desconto_pgto' => (float) $data['desconto_pgto'],
            $precificacao, 
            'pdf' => $file_name]
        );
        
        $this->conn->close();
        $this->conn = null;

    }

    /**
     * @todo Necessita de refatoração
     */
    public function render(
        string $cliente, 
        int $pessoa_juridica,
        null|string $doc_client = '',
        string $endereco_numero = '',
        string $bairro = '',
        string $cep = '',
        string $cidade = '',
        string $estado = '',
        string $pessoa_contato = '',
        string $email = '',
        string $prazo_entrega = '',
        float $valor_frete = 0,
        string $tipo_frete = '',
        array $produtos,
        float $valor_total,
        float $entrada_pgto = 0,
        int $n_parcela = 0,
        float $juros = 0,
        null|string $observacao = '',
        float $desconto_valor = 0,
    ) {
        // Instancia um novo objeto Budgets do TCPDF
        $pdf = new Budgets(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

        // Define as informações do documento
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor('Fattos');
        $pdf->SetTitle('TBT Brindes - Orçamento');
        $pdf->SetSubject('Novo orçamento');
        $pdf->SetKeywords('TCPDF, PDF, example, test, guide');

        // Define a fonte para o cabeçalho e rodapé
        $pdf->setHeaderFont(array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));

        // Define a fonte padrão monoespaçada
        $pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

        // Define as margens do documento
        $pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        $pdf->SetHeaderMargin(0);
        $pdf->SetFooterMargin(0);

        // Remove o rodapé padrão
        $pdf->setPrintFooter(false);

        // Define a quebra automática de página
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);

        // Define o fator de escala da imagem
        $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);

        // Registrar a fonte Metropolis
        $pdf->AddFont('metropolis', '', __DIR__ . '/PDFTemplates/Config/TCPDF/metropolis.php');

        // Adiciona uma página
        $pdf->AddPage();

        // Obtém a margem atual de quebra de página
        $bMargin = $pdf->getBreakMargin();

        // Obtém o modo atual de quebra automática de página
        $auto_page_break = $pdf->getAutoPageBreak();

        // Desativa a quebra automática de página
        $pdf->SetAutoPageBreak(false, 0);

        // Define a imagem de fundo
        $img_file = __DIR__ . '/../../view/assets/file/system/bgtbt.jpg';
        $pdf->Image($img_file, 0, 0, 210, 297, '', '', '', false, 300, '', false, false, 0);

        // Restaura o status de quebra automática de página
        $pdf->SetAutoPageBreak($auto_page_break, $bMargin);

        // Define o ponto inicial para o conteúdo da página
        $pdf->setPageMark();

        $pdf->SetFillColor(255, 255, 255);

        # Minha Empresa Session
        $pdf->writeHTMLCell(0, '', 15, 50, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">TBT Brindes e personalizados Ltda.</span></h1>'
            , 0, 0, 0, true, 'J', true
        );

        $pdf->writeHTMLCell(0, '', 15, 55, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">CNPJ: 36.141.688/0001-77</span></h1>'
            , 0, 0, 0, true, 'J', true
        );

        $pdf->writeHTMLCell(0, '', 15, 60, 
            '<h1><span style="color:#234A2D;font-weight:bold;font-size:9pt;">Cidade: Joinville - SC</span></h1>'
            , 0, 0, 0, true, 'J', true
        );


        # Cliente Session

        $nome_cliente = $pessoa_juridica == 1 ? 'Razão social: ' . $cliente : 'Nome: ' . $cliente;

        //Razão social ou nome
        $pdf->writeHTMLCell(0, '', 15, 75, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">'. $nome_cliente .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Dados já formatados
        $documento_cliente = $pessoa_juridica == 1 ? 'CNPJ: ' . $doc_client : 'CPF: ' . $doc_client;
        $documento_cliente = !empty($doc_client) ? $documento_cliente : 'CPF/CNPJ:';
        // CNPJ ou CPF
        $pdf->writeHTMLCell(0, '', 110, 75, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">'. $documento_cliente .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Endereço
        $pdf->writeHTMLCell(0, '', 15, 80, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Endereço: '. $endereco_numero .' </span>'
            , 0, 0, 0, true, 'J', true
        );

        // Bairro
        $pdf->writeHTMLCell(0, '', 110, 80, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Bairro: '. $bairro .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // CEP
        $pdf->writeHTMLCell(0, '', 15, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">CEP: '. $cep .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Cidade
        $pdf->writeHTMLCell(0, '', 110, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Cidade: '. $cidade .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Estado
        $pdf->writeHTMLCell(0, '', 155, 85, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Estado: '. $estado .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Contato
        $pdf->writeHTMLCell(0, '', 15, 90, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">Contato: '. $pessoa_contato .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // E-mail
        $pdf->writeHTMLCell(0, '', 110, 90, 
            '<span style="color:#234A2D;font-weight:400;font-size:9pt;">E-mail: '. $email .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // DIVISÓRIA
        $pdf->writeHTMLCell(0, '', 15, 100, 
            '<hr>'
            , 0, 0, 0, true, 'J', true
        );

        
        $product_html = '';
        $qtd_sum = 0;

        $issetPhoto = false;

        $valor_total = 0;
        foreach ($produtos as $key => $value) {
            if ($value['foto'] != null) {
                $issetPhoto = true;
            }

            $nomeCortado = strlen($value['nome']) > 37 ? htmlspecialchars(substr($value['nome'], 0, 37)) . '...' : htmlspecialchars($value['nome']);


            $qtd_produto = (int) $value['qtd_produto'];
            $valor_total_orcado = (float) $value['valor_total_orcado'];
            $valor_unitario = $valor_total_orcado / $qtd_produto;
        
            // Arredondar o valor unitário para 2 casas decimais
            $valor_unitario_arredondado = round($valor_unitario, 2);
        
            // Recalcular o valor total para garantir consistência
            $valor_total_recalculado = $valor_unitario_arredondado * $qtd_produto;

            $valor_total += $valor_total_recalculado;
        
            $qtd_sum += $qtd_produto;
            $product_html .= '<tr>';
            $product_html .= '  <td style="font-size:10pt;">'. $qtd_produto .'</td>';
            $product_html .= '  <td style="font-size:10pt;">'. $nomeCortado . '</td>';
            $product_html .= '  <td style="font-size:10pt;">'. $value['ncm']. '</td>';
            $product_html .= '  <td style="font-size:10pt;" align="right"> R$ ' . number_format($valor_unitario_arredondado, 2, ',', '.') .' &nbsp;</td>';
            $product_html .= '  <td style="font-size:10pt; text-align:right;" align="right"> R$ ' . number_format($valor_total_recalculado, 2, ',', '.') . ' &nbsp;</td>';
            $product_html .= '</tr>';
        }

        if ($juros > 0) {
            $valor_total += bcmul($valor_total, bcdiv($juros, 100, 4), 4);
        }

        if ($desconto_valor > 0) {
            $valor_total -= $desconto_valor;
        }

        if ($valor_frete > 0) {
            $valor_total += $valor_frete;
        }

        $n_parcela_info = '';
        if ($n_parcela > 0) {
            $sum = bcdiv($valor_total, $n_parcela, 4);

            $com_juros = number_format($sum, 2, ',', '.');
            $n_parcela_info = $n_parcela . 'x de R$' . $com_juros;
            $valor_total =  bcmul($sum, $n_parcela, 4);
        }

        $entrada_info = $entrada_pgto > 0 ? 'R$ ' . number_format(round($entrada_pgto, 2), 2, ',', '.') : 'Sem entrada';

        if ($entrada_pgto > 0) {
            // $valor_total += $entrada_pgto;
        }



        $valor_total_return = $valor_total;

        $frete_info   = $tipo_frete == 'FOB' ? 'Cliente retira' : 'R$ ' . number_format($valor_frete, 2, ',', '.');

        $th_desconto = '';
        $td_desconto = '';
        $cols_condicoes = 3;

        if ($desconto_valor > 0) {
            $cols_condicoes = 4;
            $th_desconto = '<th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">DESCONTO</span></th>';
            $td_desconto = '<td style="font-size:10pt;">R$ ' . number_format($desconto_valor, 2, ',', '.') . '</td>';
        }

        $html = '
            <table align="center" border="1">
                <tr>
                    <th width="60"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">QTD.</span></th>
                    <th width="290"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">DESCRIÇÃO</span></th>
                    <th width="90"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">NCM</span></th>
                    <th width="90"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR UN.</span></th>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR</span></th>
                </tr>
                '. $product_html .'
            </table>
            <br /><br /><br />

            <table align="center" border="1">
                <tr>
                    <td colspan="' . $cols_condicoes . '"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">CONDIÇÕES DE PAGAMENTO</span></td>
                </tr>
                <tr>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">ENTRADA</span></th>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">FRETE</span></th>
                    <th><span style="color:#234A2D;font-weight:bold;font-size:9pt;">PARCELAS</span></th>
                    ' . $th_desconto . '
                </tr>
                <tr>
                    <td style="font-size:10pt;">' . $entrada_info. '</td>
                    <td style="font-size:10pt;">' . $frete_info . '</td>
                    <td style="font-size:10pt;">' . $n_parcela_info .'</td>
                    '. $td_desconto .'
                </tr>
            </table>
            <br /><br /><br />
            <table align="center" border="1">
                <tr>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">QTD. TOTAL</span></th>
                    <th width="431" rowspan="2"></th>
                    <th width="100"><span style="color:#234A2D;font-weight:bold;font-size:9pt;">VALOR TOTAL</span></th>
                </tr>
                <tr>
                    <th width="100" style="font-size:10pt;">'. $qtd_sum .'</th>
                    <th width="100" style="font-size:10pt;">R$ '. number_format($valor_total, 2, ',', '.') .'</th>
                </tr>
            </table>
            <br/><br/>
            <span style="font-size:9pt;">Prazo de entrega: </span><span style="color:#234A2D;font-size:9pt;">'. $prazo_entrega .'</span>
            <br/>
            <span style="font-size:9pt;">Observações: </span><span style="color:#234A2D;font-size:9pt;"> '.$observacao.'</span>
        ';
    
        $pdf->writeHTMLCell(0, '', 15, 107, 
            $html
            , 0, 0, 0, true, 'J', true
        );

        // Data do orçamento - texto
        $pdf->writeHTMLCell(0, '', 20, 253, 
            '<span style="color:#234A2D;font-size:9pt;"><u>DATA DO ORÇAMENTO</u></span>'
            , 0, 0, 0, true, 'J', true
        );

        // Data do orçamento - data
        $pdf->writeHTMLCell(0, '', 30, 258, 
            '<span style="color:#234A2D;font-size:9pt;">'. date("d/m/Y") .'</span>'
            , 0, 0, 0, true, 'J', true
        );

        // Se não existir nenhuma foto, gera o PDF e salva
        if (!$issetPhoto) {
            $path_save = __DIR__ . '/../../view/assets/file/budgets/';
            $name_file = Helper::generaterRandomString() . date('dmy') . '.pdf';
    
            // Fecha e gera o documento PDF
            $pdf->Output($path_save . $name_file, 'F'); // save in folder
            // $pdf->Output('ae.pdf', 'I'); // preview in browser
    
            return [
                'file' => $name_file,
                'valor_total' => $valor_total_return
            ];
        }

        // ADICIONANDO NOVA PÁGINA MANUALMENTE
        $pdf->AddPage();

        // Obtém a margem atual de quebra de página
        $bMargin = $pdf->getBreakMargin();

        // Obtém o modo atual de quebra automática de página
        $auto_page_break = $pdf->getAutoPageBreak();

        // Desativa a quebra automática de página
        $pdf->SetAutoPageBreak(false, 0);

        $pdf->Image($img_file, 0, 0, 210, 297, '', '', '', false, 300, '', false, false, 0);

        // Restaura o status de quebra automática de página
        $pdf->SetAutoPageBreak($auto_page_break, $bMargin);

        // Define o ponto inicial para o conteúdo da página
        $pdf->setPageMark();

        $pdf->SetFillColor(255, 255, 255);

        // Inicia a posição y onde as imagens serão colocadas
        $y = 55;

        // Inicia a posição x onde as imagens serão colocadas
        $x = 15;

        // Contador para as imagens
        $contador = 0;

        // Itera sobre o array de imagens
        foreach ($produtos as $key => $value) {

            // Se não houver uma imagem, pula para a próxima
            if ($value['foto'] == null) {
                continue;
            }
        
            $file_url = __DIR__ . '/../../view/assets/file/product/' . $value['foto'];
        
            // Verifica se o arquivo de imagem existe
            if (!file_exists($file_url)) {
                continue;
            }
        
            $name_product = strlen($value['nome']) > 25 ? mb_substr($value['nome'], 0, 25, 'UTF-8') . '...' : $value['nome'];
            
            // Adiciona o nome do produto ao PDF
            $pdf->writeHTMLCell(0, '', $x, $y, 
                '<span style="color:#234A2D;font-weight:bold;font-size:9pt;">'. $name_product .'</span>', 
                0, 0, 0, true, 'J', true
            );
        
            // Adiciona a imagem ao PDF
            $pdf->Image($file_url, $x, $y + 5, 48, 48, '', '', '', true, 150, '', false, false, 0, false, false, false);
        
            // Incrementa o contador
            $contador++;
        
            // Se o contador é divisível por 3, inicia uma nova linha
            if ($contador % 3 == 0) {
                $y += 60; // 60 para a altura da imagem e 10 para o espaço entre as linhas
                $x = 15; // Reinicia a posição x para a margem esquerda
            } else {
                $x += 60; // 60 para a largura da imagem e 10 para o espaço entre as imagens
            }
        
            // Se o contador é divisível por 9, adiciona uma nova página
            if ($contador % 9 == 0) {
                $pdf->AddPage();
                // Reinicia a posição y para a margem superior
                $y = 55;
            }
        }


        $path_save = __DIR__ . '/../../view/assets/file/budgets/';
        $name_file = Helper::generaterRandomString() . date('dmy') . '.pdf';

        // Fecha e gera o documento PDF
        $pdf->Output($path_save . $name_file, 'F'); // save in folder
        // $pdf->Output('ae.pdf', 'I'); // preview in browser

        return [
            'file' => $name_file,
            'valor_total' => $valor_total_return
        ];
    }

    public function getCountBudgets(null|string $like = ''): int
    {
        $tblOrcamentos = RenderTable::table('orcamentos');
        $tblClientes = RenderTable::table('clientes');
        $tblUsuarios = RenderTable::table('usuarios');

        $where = $like ? ' ' . $like : ' ' . $like;

        $query = 
            "SELECT 
                COUNT(*) AS qtd 
            FROM $tblOrcamentos AS o
            LEFT JOIN $tblClientes AS c ON c.id = o.id_cliente_FK
            LEFT JOIN $tblUsuarios as u ON u.id = o.create_by
            WHERE 1 = 1 $where";

        $data = self::preparedQuery($query);
        return !empty($data[0]['qtd']) ? (int)$data[0]['qtd'] : 0;
    }

    private function filterOrcamentoLike(
        null|string $filter = '',
        null|string $status = '',
        null|string $retorno = '',
        null|string $vendedor = '',
        $clienteId = '',
        ?int $orcamentoId = null
    ) {

        $sql_like = $filter ? " AND (c.nome LIKE '%$filter%' OR u.nome LIKE '%$filter%' OR o.id LIKE '%$filter%')" : '';

        if ($vendedor) {
            $sql_like .= " AND o.create_by = $vendedor";
        }

        if ($clienteId) {
            $sql_like .= " AND o.id_cliente_FK = $clienteId";
        }

        if ($status) {
            $sql_like .= " AND o.status_user = $status";
        }

        if ($orcamentoId) {
            $sql_like .= " AND o.id = $orcamentoId";
        }

        if ($retorno) {

            switch ($retorno) {
                case 'this': // Hoje
                    $retorno = ' = CURDATE()';
                    break;
                case 'next': // Amanhã
                    $retorno = ' = DATE_ADD(CURDATE(), INTERVAL 1 DAY)';
                    break;
                case 'last': // Atrasado
                    $retorno = " < CURDATE() AND o.data_retorno >= '" . $this->dataDinamica . "'";
                    break;
                case 'future': // Futuro
                    $retorno = " > DATE_ADD(CURDATE(), INTERVAL 2 DAY)";
                    break; 
                default:
                    break;
            }

            $sql_like .= " AND o.data_retorno $retorno";
        }
       
        return $sql_like;
    }


    /**
     * ! @TODO PROBLEMA DE SEGURANÇA
     */
    public function showList(int $current_page, int $limit_per_page, null|string $filter = ''): array
    {
        try {
            $this->conn  = Conn::openConn();

            $tblOrcamentos = RenderTable::table('orcamentos');
            $tblClientes = RenderTable::table('clientes');
            $tblUsuarios = RenderTable::table('usuarios');
            $tblStatus = RenderTable::table('status');

            $getStatus = isset($_GET['status']) ? $_GET['status'] : '';
            $getRetorno = isset($_GET['retorno']) ? $_GET['retorno'] : '';
            $getVendedor = isset($_GET['vendedor']) ? $_GET['vendedor'] : '';
            $getClienteId = isset($_GET['clienteId']) ? $_GET['clienteId'] : '';
            $getOrcamentoId = isset($_GET['orcamentoId']) ? (int)$_GET['orcamentoId'] : null;

            $sql_like = $this->filterOrcamentoLike($filter ?? '', $getStatus, $getRetorno, $getVendedor, $getClienteId, $getOrcamentoId);

            $paginate = new Paginate($this->getCountBudgets($sql_like), $current_page, $limit_per_page);

            $where_vendedor = $_SESSION['user']['nivel'] < '3' 
                ? "AND (o.create_by = {$_SESSION['user']['id']} OR c.created_by = {$_SESSION['user']['id']})" 
                : '';

            $getDiasPerder = (new Config)->getUltimoDiaTempoPerda();
            $this->dataDinamica  = date('Y-m-d', strtotime("-$getDiasPerder days"));

            $SQL = 
                "SELECT
                    o.id,
                    o.dialogo,
                    o.valor_total,
                    c.nome AS cliente,
                    CONCAT(SUBSTRING_INDEX(u.nome, ' ', 1), ' ', SUBSTRING_INDEX(u.nome, ' ', -1)) AS nome,
                    u.email,
                    o.desconto_status,
                    o.status_user,
                    s.nome AS nomeStatus,
                    u.img AS img_usuario,
                    DATE_FORMAT(o.data_retorno, '%d/%m/%Y') AS dataRetornoBr,
                    DATE_FORMAT(o.create_at, '%H:%i %d/%m/%Y') AS data_orcado,
                    DATE_FORMAT(o.data_retorno, '%d/%m/%Y') AS data_retorno,
                    CASE
                        WHEN o.status_user > 24 AND o.status_user < 29 THEN 0
                        WHEN o.status_user = 21 THEN 1
                        WHEN o.data_retorno = CURDATE() THEN 'Hoje'
                        WHEN o.data_retorno = DATE_ADD(CURDATE(), INTERVAL 1 DAY) THEN 'Amanhã' -- #ffcc00
                        WHEN o.data_retorno < CURDATE() AND o.data_retorno >= '{$this->dataDinamica}' THEN 'Retorno atrasado' -- #91030a
                        WHEN o.data_retorno < '{$this->dataDinamica}' THEN -1
                        ELSE DATE_FORMAT(o.data_retorno, '%d/%m/%Y')
                    END AS statusSistema
                FROM $tblOrcamentos AS o
                LEFT JOIN $tblClientes AS c ON c.id = o.id_cliente_FK
                LEFT JOIN $tblUsuarios as u ON u.id = o.create_by
                LEFT JOIN $tblStatus as s ON s.id = o.status_user
                WHERE 1 = 1 
                $where_vendedor
                $sql_like
                GROUP BY o.id, o.data_retorno, c.nome, o.status
                ORDER BY o.id DESC
                LIMIT {$paginate->getLimitOffset()}
            ";

            $data = self::PreparedQuery($SQL);
    
            $this->conn->close();
            $this->conn = null;
    
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }

        return [
            'data' => $data,
            'pages' => $paginate->generatePaginationData($current_page)
        ]; 
    }

    public function show()
    {
        # ---------------------------------
        # Status
        # -1 = Perdeu (Laranja)
        # 0 = Stand-by (Coloração por data)
        # 1 = Ganhou (Verde)
        # desconto = -1 (preto)
        # ---------------------------------

        # ---------------------------------
        # Data de retorno
        # Amarelo: Na data de retorno ou um dia antes da data de retorno 
        # Vermelho: Atrasado
        # Azul: Em orçamento (Tem folga em dias até a data de retorno)
        # Preto: Desconto sem tratativa pelo superior
        # ---------------------------------

       // OBTER DIAS PERDA AUTOMATICA E JOGAR EM UMA VARIÁVEL 
 
       $dias = (new Config)->getUltimoDiaTempoPerda();

        try {
            $this->conn  = Conn::openConn();
            
            $tblOrcamentos = RenderTable::table('orcamentos');
            $tblClientes = RenderTable::table('clientes');

            $where_vendedor = $_SESSION['user']['nivel'] < 3 ? "AND o.create_by = {$_SESSION['user']['id']}" : '';

            $SQL  = 
                "SELECT
                    o.id,
                    o.data_retorno AS start,
                    CONCAT('#', o.id, ', ', c.nome) AS title,
                    '#c3c3c3' AS borderColor,
                    CASE
                        WHEN o.status = -1 THEN '#e04a04' -- Se perdido
                        WHEN o.data_retorno < DATE_SUB(CURDATE(), INTERVAL $dias DAY) THEN '#e04a04' -- Se a data de retorno estiver passado a quantidade de dias informada para perder automaticamente pelo usuário
                        WHEN o.desconto_status = -1 THEN '#000' -- Se solicitação de desconto não estiver sido respondida
                        WHEN o.status = 1 THEN '#206e09' -- Se ganhou
                        WHEN o.data_retorno = CURDATE() OR o.data_retorno = DATE_ADD(CURDATE(), INTERVAL 1 DAY) THEN '#ffcc00' -- Se estiver no dia ou faltando um dia para o retorno do orçamento
                        WHEN o.data_retorno < CURDATE() THEN '#91030a' -- Se está atrasado o retorno do orçamento
                        WHEN o.data_retorno < DATE_SUB(CURDATE(), INTERVAL 2 DAY) THEN '#0000FF' -- Se estiver tempo de folga para retornar o orçammento
                        ELSE NULL
                    END AS backgroundColor,
                    CASE 
                        WHEN -- Se solicitação de desconto não respondida e não ultrapassar o dia da perda
                            o.desconto_status = -1 
                            AND o.data_retorno > DATE_SUB(CURDATE(), INTERVAL $dias DAY)
                            THEN '1'
                        ELSE NULL
                    END AS isUrgent
                FROM $tblOrcamentos AS o
                LEFT JOIN $tblClientes AS c ON c.id = o.id_cliente_FK
                WHERE 1 = 1 $where_vendedor
                GROUP BY o.id, o.data_retorno, c.nome, o.status;
            ";

            // Helper::jsonResponse($SQL, 400);

            $data = self::PreparedQuery($SQL);
    
            $this->conn->close();
            $this->conn = null;
    
        } catch (\Throwable $th) {
            Helper::jsonResponse($th->getMessage(), 400);
        }

        return $data; 
    }

    /**
     * Método responsável por atualizar o status do orçamento (1: ganho, -1: perdido);
     * 
     * @param int $id_orcamento 
     * @param int $status
     */
    public function toggleStatus(int $id_orcamento, int $status)
    {
        $this->conn  = Conn::openConn();

        self::updateQuery(
            RenderTable::table('orcamentos'), 
            ['status' => $status], 
            ['id' => $id_orcamento]
        );

        $this->conn->close();
        $this->conn = null;

        Helper::jsonResponse(1);
    }

    /**
     * Método responsável por retornar dados do orçamento para visualização
     * 
     * @param int $id_orcamento - chave primária do orçamento
     * 
     * @return array $data
     * 
     * @todo Implementar captura das informações de diálogo 
     * @todo Implementar no select código do produto (provávelmente terá de fazer um join)
     */
    public function viewBudget(int $id_orcamento): array
    {
        $this->conn  = Conn::openConn();
        $tblOrcamentos = RenderTable::table('orcamentos');
        $tblClientes   = RenderTable::table('clientes');

        $SQL = 
            "SELECT 
                o.vezes_juros, 
                o.juros_porcentagem, 
                o.observacao_pedido, 
                o.prazo_entrega, 
                o.frete_tipo, 
                o.entrada_pgto, 
                o.valor_total, 
                o.dialogo, 
                o.file, 
                o.data_retorno, 
                o.create_at, 
                o.pessoa_contato, 
                o.status, 
                c.nome,
                o.desconto_status,
                o.desconto_valor
            FROM $tblOrcamentos AS o
            LEFT JOIN $tblClientes AS c
            ON c.id = o.id_cliente_FK
            WHERE o.id = ?
        ";

        $orcamento = self::PreparedQuery($SQL, [$id_orcamento]);

        $tblOrcamentosProdutos = RenderTable::table('orcamentos_produtos');
        $tblProdutos = RenderTable::table('produtos');

        $SQL = 
            "SELECT 
                p.nome, 
                op.qtd, 
                op.valor_total_produto, 
                op.precificacao_nome
            FROM $tblOrcamentosProdutos AS op
            LEFT JOIN $tblProdutos AS p
            ON p.id = op.id_produto_FK
            INNER JOIN (
                SELECT id_orcamento_FK, MAX(id_orcamento_historico_FK) AS max_id
                FROM $tblOrcamentosProdutos
                GROUP BY id_orcamento_FK
            ) AS subquery
            ON op.id_orcamento_FK = subquery.id_orcamento_FK AND op.id_orcamento_historico_FK = subquery.max_id
            WHERE op.id_orcamento_FK = ?
        ";
    

        $produtos = self::PreparedQuery($SQL, [$id_orcamento]);

        $this->conn->close();
        $this->conn = null;

        return [
            'orcamento_info' => $orcamento[0],
            'produtos' => $produtos,
            'is_admin' => $_SESSION['user']['nivel'] > 7
        ]; 
    }

    public function updateBudget(array $data)
    {
        // Obter o id do orçamento
        
        $this->isUpdate = true;
        $res = $this->store($data, false);
        Helper::jsonResponse($res);

        // Necessário chamar método para guardar no histórico
        // Ao guardar no histório, precisarei do id do histórico
        // Para posteriormente associar os produtos dessa alteração
        // na tabela orcamentos_produtos
        // Renderizar um novo PDF será necessário
        // Não deverei excluir o PDF antigo
        // Criar forma de versionar os PDFs
        // Visualizar integridade dos dados posteriormente
        // Verificar se os dados estão sendo salvos corretamente
    }

    /**
     * Método responsável por retornar a função solicitada pelo front-end
     * 
     * @param string $route
     * 
     * @return void
     */
    private function route(string $route): void
    {
        $auth_middleware = new AuthenticationMiddleware;
        
        match ($route) {
            'store' => $auth_middleware->handle(
                fn() => $this->store($_REQUEST),
            ),
            'preview' => $auth_middleware->handle(
                fn() => $this->store($_REQUEST, true),
            ),
            'show' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->show()),
            ),
            'viewBudget' => $auth_middleware->handle(
                fn() => Helper::jsonResponse($this->viewBudget($_POST['id']))
            ),
            'toggle' => $auth_middleware->handle(
                fn() => $this->toggleStatus((int)$_POST['id'], (int)$_POST['status'])
            ),
            'update' => $auth_middleware->handle(
                fn() => $this->updateBudget($_REQUEST)
            )
        };
    }

    public function setRoute(string $route): void
    {
        $this->route($route);
    }
}

if (isset($_POST['action']) && Helper::validateRequest($_SERVER['REQUEST_URI']) == 'Budgets') 
{
    $instance = new Budgets();
    $instance->setRoute($_REQUEST['action']);
}
